/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.hystrix.util;

import com.netflix.hystrix.strategy.properties.HystrixProperty;
import com.netflix.hystrix.util.HystrixRollingNumberEvent;
import com.netflix.hystrix.util.LongAdder;
import com.netflix.hystrix.util.LongMaxUpdater;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.concurrent.ThreadSafe;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class HystrixRollingNumber {
    private static final Logger logger = LoggerFactory.getLogger(HystrixRollingNumber.class);
    private static final Time ACTUAL_TIME = new ActualTime();
    private final Time time;
    private final HystrixProperty<Integer> timeInMilliseconds;
    private final HystrixProperty<Integer> numberOfBuckets;
    private final BucketCircularArray buckets;
    private final CumulativeSum cumulativeSum = new CumulativeSum();
    private ReentrantLock newBucketLock = new ReentrantLock();

    public HystrixRollingNumber(HystrixProperty<Integer> timeInMilliseconds, HystrixProperty<Integer> numberOfBuckets) {
        this(ACTUAL_TIME, timeInMilliseconds, numberOfBuckets);
    }

    private HystrixRollingNumber(Time time, int timeInMilliseconds, int numberOfBuckets) {
        this(time, HystrixProperty.Factory.asProperty(timeInMilliseconds), HystrixProperty.Factory.asProperty(numberOfBuckets));
    }

    private HystrixRollingNumber(Time time, HystrixProperty<Integer> timeInMilliseconds, HystrixProperty<Integer> numberOfBuckets) {
        this.time = time;
        this.timeInMilliseconds = timeInMilliseconds;
        this.numberOfBuckets = numberOfBuckets;
        if (timeInMilliseconds.get() % numberOfBuckets.get() != 0) {
            throw new IllegalArgumentException("The timeInMilliseconds must divide equally into numberOfBuckets. For example 1000/10 is ok, 1000/11 is not.");
        }
        this.buckets = new BucketCircularArray(numberOfBuckets.get());
    }

    private int getBucketSizeInMilliseconds() {
        return this.timeInMilliseconds.get() / this.numberOfBuckets.get();
    }

    public void increment(HystrixRollingNumberEvent type) {
        this.getCurrentBucket().getAdder(type).increment();
    }

    public void add(HystrixRollingNumberEvent type, long value) {
        this.getCurrentBucket().getAdder(type).add(value);
    }

    public void updateRollingMax(HystrixRollingNumberEvent type, long value) {
        this.getCurrentBucket().getMaxUpdater(type).update(value);
    }

    public void reset() {
        Bucket lastBucket = this.buckets.peekLast();
        if (lastBucket != null) {
            this.cumulativeSum.addBucket(lastBucket);
        }
        this.buckets.clear();
    }

    public long getCumulativeSum(HystrixRollingNumberEvent type) {
        return this.getValueOfLatestBucket(type) + this.cumulativeSum.get(type);
    }

    public long getRollingSum(HystrixRollingNumberEvent type) {
        Bucket lastBucket = this.getCurrentBucket();
        if (lastBucket == null) {
            return 0L;
        }
        long sum = 0L;
        for (Bucket b : this.buckets) {
            sum += b.getAdder(type).sum();
        }
        return sum;
    }

    public long getValueOfLatestBucket(HystrixRollingNumberEvent type) {
        Bucket lastBucket = this.getCurrentBucket();
        if (lastBucket == null) {
            return 0L;
        }
        return lastBucket.get(type);
    }

    public long[] getValues(HystrixRollingNumberEvent type) {
        Bucket lastBucket = this.getCurrentBucket();
        if (lastBucket == null) {
            return new long[0];
        }
        Bucket[] bucketArray = this.buckets.getArray();
        long[] values = new long[bucketArray.length];
        int i = 0;
        for (Bucket bucket : bucketArray) {
            if (type.isCounter()) {
                values[i++] = bucket.getAdder(type).sum();
                continue;
            }
            if (!type.isMaxUpdater()) continue;
            values[i++] = bucket.getMaxUpdater(type).max();
        }
        return values;
    }

    public long getRollingMaxValue(HystrixRollingNumberEvent type) {
        long[] values = this.getValues(type);
        if (values.length == 0) {
            return 0L;
        }
        Arrays.sort(values);
        return values[values.length - 1];
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Bucket getCurrentBucket() {
        long currentTime = this.time.getCurrentTimeInMillis();
        Bucket currentBucket = this.buckets.peekLast();
        if (currentBucket != null && currentTime < currentBucket.windowStart + (long)this.getBucketSizeInMilliseconds()) {
            return currentBucket;
        }
        if (this.newBucketLock.tryLock()) {
            try {
                if (this.buckets.peekLast() == null) {
                    Bucket newBucket = new Bucket(currentTime);
                    this.buckets.addLast(newBucket);
                    Bucket bucket = newBucket;
                    return bucket;
                }
                for (int i = 0; i < this.numberOfBuckets.get(); ++i) {
                    Bucket lastBucket = this.buckets.peekLast();
                    if (currentTime < lastBucket.windowStart + (long)this.getBucketSizeInMilliseconds()) {
                        Bucket bucket = lastBucket;
                        return bucket;
                    }
                    if (currentTime - (lastBucket.windowStart + (long)this.getBucketSizeInMilliseconds()) > (long)this.timeInMilliseconds.get().intValue()) {
                        this.reset();
                        Bucket bucket = this.getCurrentBucket();
                        return bucket;
                    }
                    this.buckets.addLast(new Bucket(lastBucket.windowStart + (long)this.getBucketSizeInMilliseconds()));
                    this.cumulativeSum.addBucket(lastBucket);
                }
                Bucket i = this.buckets.peekLast();
                return i;
            }
            finally {
                this.newBucketLock.unlock();
            }
        }
        currentBucket = this.buckets.peekLast();
        if (currentBucket != null) {
            return currentBucket;
        }
        try {
            Thread.sleep(5L);
        }
        catch (Exception e) {
            // empty catch block
        }
        return this.getCurrentBucket();
    }

    public static class UnitTest {
        @Test
        public void testCreatesBuckets() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                Assert.assertEquals((long)200L, (long)((Integer)counter.timeInMilliseconds.get()).intValue());
                Assert.assertEquals((long)10L, (long)((Integer)counter.numberOfBuckets.get()).intValue());
                Assert.assertEquals((long)20L, (long)counter.getBucketSizeInMilliseconds());
                Assert.assertEquals((long)0L, (long)counter.buckets.size());
                for (int i = 0; i < (Integer)counter.numberOfBuckets.get(); ++i) {
                    counter.increment(HystrixRollingNumberEvent.SUCCESS);
                    time.increment(counter.getBucketSizeInMilliseconds());
                }
                Assert.assertEquals((long)10L, (long)counter.buckets.size());
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)10L, (long)counter.buckets.size());
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testResetBuckets() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                Assert.assertEquals((long)0L, (long)counter.buckets.size());
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testEmptyBucketsFillIn() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testIncrementInSingleBucket() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)4L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.SUCCESS).sum());
                Assert.assertEquals((long)2L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.FAILURE).sum());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.TIMEOUT).sum());
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testTimeout() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.TIMEOUT).sum());
                Assert.assertEquals((long)1L, (long)counter.getRollingSum(HystrixRollingNumberEvent.TIMEOUT));
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.TIMEOUT).sum());
                Assert.assertEquals((long)2L, (long)counter.getRollingSum(HystrixRollingNumberEvent.TIMEOUT));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testShortCircuited() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.SHORT_CIRCUITED);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.SHORT_CIRCUITED).sum());
                Assert.assertEquals((long)1L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SHORT_CIRCUITED));
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.increment(HystrixRollingNumberEvent.SHORT_CIRCUITED);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.SHORT_CIRCUITED).sum());
                Assert.assertEquals((long)2L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SHORT_CIRCUITED));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testThreadPoolRejection() {
            this.testCounterType(HystrixRollingNumberEvent.THREAD_POOL_REJECTED);
        }

        @Test
        public void testFallbackSuccess() {
            this.testCounterType(HystrixRollingNumberEvent.FALLBACK_SUCCESS);
        }

        @Test
        public void testFallbackFailure() {
            this.testCounterType(HystrixRollingNumberEvent.FALLBACK_FAILURE);
        }

        @Test
        public void testExceptionThrow() {
            this.testCounterType(HystrixRollingNumberEvent.EXCEPTION_THROWN);
        }

        private void testCounterType(HystrixRollingNumberEvent type) {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(type);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(type).sum());
                Assert.assertEquals((long)1L, (long)counter.getRollingSum(type));
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.increment(type);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(type).sum());
                Assert.assertEquals((long)2L, (long)counter.getRollingSum(type));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testIncrementInMultipleBuckets() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                counter.increment(HystrixRollingNumberEvent.SHORT_CIRCUITED);
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.TIMEOUT);
                counter.increment(HystrixRollingNumberEvent.SHORT_CIRCUITED);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)2L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.SUCCESS).sum());
                Assert.assertEquals((long)3L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.FAILURE).sum());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.TIMEOUT).sum());
                Assert.assertEquals((long)1L, (long)counter.buckets.getLast().getAdder(HystrixRollingNumberEvent.SHORT_CIRCUITED).sum());
                Assert.assertEquals((long)6L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS));
                Assert.assertEquals((long)5L, (long)counter.getRollingSum(HystrixRollingNumberEvent.FAILURE));
                Assert.assertEquals((long)3L, (long)counter.getRollingSum(HystrixRollingNumberEvent.TIMEOUT));
                Assert.assertEquals((long)2L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SHORT_CIRCUITED));
                time.increment((Integer)counter.timeInMilliseconds.get());
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)1L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS));
                Assert.assertEquals((long)0L, (long)counter.getRollingSum(HystrixRollingNumberEvent.FAILURE));
                Assert.assertEquals((long)0L, (long)counter.getRollingSum(HystrixRollingNumberEvent.TIMEOUT));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testCounterRetrievalRefreshesBuckets() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                counter.increment(HystrixRollingNumberEvent.FAILURE);
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)4L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS));
                Assert.assertEquals((long)2L, (long)counter.getRollingSum(HystrixRollingNumberEvent.FAILURE));
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                time.increment((Integer)counter.timeInMilliseconds.get());
                Assert.assertEquals((long)0L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS));
                Assert.assertEquals((long)0L, (long)counter.getRollingSum(HystrixRollingNumberEvent.FAILURE));
                counter.increment(HystrixRollingNumberEvent.SUCCESS);
                Assert.assertEquals((long)1L, (long)counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS));
                Assert.assertEquals((long)0L, (long)counter.getRollingSum(HystrixRollingNumberEvent.FAILURE));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testUpdateMax1() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 10L);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)10L, (long)counter.buckets.getLast().getMaxUpdater(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE).max());
                Assert.assertEquals((long)10L, (long)counter.getRollingMaxValue(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE));
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 20L);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)20L, (long)counter.buckets.getLast().getMaxUpdater(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE).max());
                long[] values = counter.getValues(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE);
                Assert.assertEquals((long)10L, (long)values[0]);
                Assert.assertEquals((long)0L, (long)values[1]);
                Assert.assertEquals((long)0L, (long)values[2]);
                Assert.assertEquals((long)20L, (long)values[3]);
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testUpdateMax2() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 10L);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 30L);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 20L);
                Assert.assertEquals((long)1L, (long)counter.buckets.size());
                Assert.assertEquals((long)30L, (long)counter.buckets.getLast().getMaxUpdater(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE).max());
                Assert.assertEquals((long)30L, (long)counter.getRollingMaxValue(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE));
                time.increment(counter.getBucketSizeInMilliseconds() * 3);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 30L);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 30L);
                counter.updateRollingMax(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE, 50L);
                Assert.assertEquals((long)4L, (long)counter.buckets.size());
                Assert.assertEquals((long)50L, (long)counter.buckets.getLast().getMaxUpdater(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE).max());
                Assert.assertEquals((long)50L, (long)counter.getValueOfLatestBucket(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE));
                long[] values = counter.getValues(HystrixRollingNumberEvent.THREAD_MAX_ACTIVE);
                Assert.assertEquals((long)30L, (long)values[0]);
                Assert.assertEquals((long)0L, (long)values[1]);
                Assert.assertEquals((long)0L, (long)values[2]);
                Assert.assertEquals((long)50L, (long)values[3]);
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testMaxValue() {
            MockedTime time = new MockedTime();
            try {
                HystrixRollingNumberEvent type = HystrixRollingNumberEvent.THREAD_MAX_ACTIVE;
                HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
                counter.updateRollingMax(type, 10L);
                time.increment(counter.getBucketSizeInMilliseconds());
                counter.updateRollingMax(type, 30L);
                time.increment(counter.getBucketSizeInMilliseconds());
                counter.updateRollingMax(type, 40L);
                time.increment(counter.getBucketSizeInMilliseconds());
                counter.updateRollingMax(type, 15L);
                Assert.assertEquals((long)40L, (long)counter.getRollingMaxValue(type));
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Exception: " + e.getMessage()));
            }
        }

        @Test
        public void testEmptySum() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.COLLAPSED;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
            Assert.assertEquals((long)0L, (long)counter.getRollingSum(type));
        }

        @Test
        public void testEmptyMax() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.THREAD_MAX_ACTIVE;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
            Assert.assertEquals((long)0L, (long)counter.getRollingMaxValue(type));
        }

        @Test
        public void testEmptyLatestValue() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.THREAD_MAX_ACTIVE;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 200, 10);
            Assert.assertEquals((long)0L, (long)counter.getValueOfLatestBucket(type));
        }

        @Test
        public void testRolling() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.THREAD_MAX_ACTIVE;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 20, 2);
            for (int i = 0; i < 20; ++i) {
                counter.getCurrentBucket();
                try {
                    time.increment(counter.getBucketSizeInMilliseconds());
                }
                catch (Exception e) {
                    // empty catch block
                }
                Assert.assertEquals((long)2L, (long)counter.getValues(type).length);
                counter.getValueOfLatestBucket(type);
            }
        }

        @Test
        public void testCumulativeCounterAfterRolling() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.SUCCESS;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 20, 2);
            Assert.assertEquals((long)0L, (long)counter.getCumulativeSum(type));
            for (int i = 0; i < 20; ++i) {
                counter.increment(type);
                try {
                    time.increment(counter.getBucketSizeInMilliseconds());
                }
                catch (Exception e) {
                    // empty catch block
                }
                Assert.assertEquals((long)2L, (long)counter.getValues(type).length);
                counter.getValueOfLatestBucket(type);
            }
            Assert.assertEquals((long)20L, (long)counter.getCumulativeSum(type));
        }

        @Test
        public void testCumulativeCounterAfterRollingAndReset() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.SUCCESS;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 20, 2);
            Assert.assertEquals((long)0L, (long)counter.getCumulativeSum(type));
            for (int i = 0; i < 20; ++i) {
                counter.increment(type);
                try {
                    time.increment(counter.getBucketSizeInMilliseconds());
                }
                catch (Exception e) {
                    // empty catch block
                }
                Assert.assertEquals((long)2L, (long)counter.getValues(type).length);
                counter.getValueOfLatestBucket(type);
                if (i != 5 && i != 15) continue;
                counter.reset();
            }
            Assert.assertEquals((long)20L, (long)counter.getCumulativeSum(type));
        }

        @Test
        public void testCumulativeCounterAfterRollingAndReset2() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.SUCCESS;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 20, 2);
            Assert.assertEquals((long)0L, (long)counter.getCumulativeSum(type));
            counter.increment(type);
            counter.increment(type);
            counter.increment(type);
            for (int i = 0; i < 20; ++i) {
                try {
                    time.increment(counter.getBucketSizeInMilliseconds());
                }
                catch (Exception e) {
                    // empty catch block
                }
                if (i != 5 && i != 15) continue;
                counter.reset();
            }
            counter.increment(type);
            counter.increment(type);
            Assert.assertEquals((long)5L, (long)counter.getCumulativeSum(type));
        }

        @Test
        public void testCumulativeCounterAfterRollingAndReset3() {
            MockedTime time = new MockedTime();
            HystrixRollingNumberEvent type = HystrixRollingNumberEvent.SUCCESS;
            HystrixRollingNumber counter = new HystrixRollingNumber(time, 20, 2);
            Assert.assertEquals((long)0L, (long)counter.getCumulativeSum(type));
            counter.increment(type);
            counter.increment(type);
            counter.increment(type);
            for (int i = 0; i < 20; ++i) {
                try {
                    time.increment(counter.getBucketSizeInMilliseconds());
                    continue;
                }
                catch (Exception e) {
                    // empty catch block
                }
            }
            counter.increment(type);
            counter.increment(type);
            Assert.assertEquals((long)5L, (long)counter.getCumulativeSum(type));
        }

        private static class MockedTime
        implements Time {
            private AtomicInteger time = new AtomicInteger(0);

            private MockedTime() {
            }

            @Override
            public long getCurrentTimeInMillis() {
                return this.time.get();
            }

            public void increment(int millis) {
                this.time.addAndGet(millis);
            }
        }
    }

    private class BucketCircularArray
    implements Iterable<Bucket> {
        private final AtomicReference<ListState> state;
        private final int dataLength;
        private final int numBuckets;

        BucketCircularArray(int size) {
            AtomicReferenceArray _buckets = new AtomicReferenceArray(size + 1);
            this.state = new AtomicReference<ListState>(new ListState(_buckets, 0, 0));
            this.dataLength = _buckets.length();
            this.numBuckets = size;
        }

        public void clear() {
            ListState newState;
            ListState current;
            while (!this.state.compareAndSet(current = this.state.get(), newState = current.clear())) {
            }
        }

        @Override
        public Iterator<Bucket> iterator() {
            return Collections.unmodifiableList(Arrays.asList(this.getArray())).iterator();
        }

        public void addLast(Bucket o) {
            ListState newState;
            ListState currentState = this.state.get();
            if (this.state.compareAndSet(currentState, newState = currentState.addBucket(o))) {
                return;
            }
        }

        public Bucket getLast() {
            return this.peekLast();
        }

        public int size() {
            return this.state.get().size;
        }

        public Bucket peekLast() {
            return this.state.get().tail();
        }

        private Bucket[] getArray() {
            return this.state.get().getArray();
        }

        private class ListState {
            private final AtomicReferenceArray<Bucket> data;
            private final int size;
            private final int tail;
            private final int head;

            private ListState(AtomicReferenceArray<Bucket> data, int head, int tail) {
                this.head = head;
                this.tail = tail;
                this.size = head == 0 && tail == 0 ? 0 : (tail + BucketCircularArray.this.dataLength - head) % BucketCircularArray.this.dataLength;
                this.data = data;
            }

            public Bucket tail() {
                if (this.size == 0) {
                    return null;
                }
                return this.data.get(this.convert(this.size - 1));
            }

            private Bucket[] getArray() {
                ArrayList<Bucket> array = new ArrayList<Bucket>();
                for (int i = 0; i < this.size; ++i) {
                    array.add(this.data.get(this.convert(i)));
                }
                return array.toArray(new Bucket[array.size()]);
            }

            private ListState incrementTail() {
                if (this.size == BucketCircularArray.this.numBuckets) {
                    return new ListState(this.data, (this.head + 1) % BucketCircularArray.this.dataLength, (this.tail + 1) % BucketCircularArray.this.dataLength);
                }
                return new ListState(this.data, this.head, (this.tail + 1) % BucketCircularArray.this.dataLength);
            }

            public ListState clear() {
                return new ListState(new AtomicReferenceArray<Bucket>(BucketCircularArray.this.dataLength), 0, 0);
            }

            public ListState addBucket(Bucket b) {
                this.data.set(this.tail, b);
                return this.incrementTail();
            }

            private int convert(int index) {
                return (index + this.head) % BucketCircularArray.this.dataLength;
            }
        }
    }

    private static class CumulativeSum {
        final LongAdder[] adderForCounterType = new LongAdder[HystrixRollingNumberEvent.values().length];
        final LongMaxUpdater[] updaterForCounterType;

        CumulativeSum() {
            for (HystrixRollingNumberEvent type : HystrixRollingNumberEvent.values()) {
                if (!type.isCounter()) continue;
                this.adderForCounterType[type.ordinal()] = new LongAdder();
            }
            this.updaterForCounterType = new LongMaxUpdater[HystrixRollingNumberEvent.values().length];
            for (HystrixRollingNumberEvent type : HystrixRollingNumberEvent.values()) {
                if (!type.isMaxUpdater()) continue;
                this.updaterForCounterType[type.ordinal()] = new LongMaxUpdater();
                this.updaterForCounterType[type.ordinal()].update(0L);
            }
        }

        public void addBucket(Bucket lastBucket) {
            for (HystrixRollingNumberEvent type : HystrixRollingNumberEvent.values()) {
                if (type.isCounter()) {
                    this.getAdder(type).add(lastBucket.getAdder(type).sum());
                }
                if (!type.isMaxUpdater()) continue;
                this.getMaxUpdater(type).update(lastBucket.getMaxUpdater(type).max());
            }
        }

        long get(HystrixRollingNumberEvent type) {
            if (type.isCounter()) {
                return this.adderForCounterType[type.ordinal()].sum();
            }
            if (type.isMaxUpdater()) {
                return this.updaterForCounterType[type.ordinal()].max();
            }
            throw new IllegalStateException("Unknown type of event: " + type.name());
        }

        LongAdder getAdder(HystrixRollingNumberEvent type) {
            if (!type.isCounter()) {
                throw new IllegalStateException("Type is not a Counter: " + type.name());
            }
            return this.adderForCounterType[type.ordinal()];
        }

        LongMaxUpdater getMaxUpdater(HystrixRollingNumberEvent type) {
            if (!type.isMaxUpdater()) {
                throw new IllegalStateException("Type is not a MaxUpdater: " + type.name());
            }
            return this.updaterForCounterType[type.ordinal()];
        }
    }

    private static class Bucket {
        final long windowStart;
        final LongAdder[] adderForCounterType;
        final LongMaxUpdater[] updaterForCounterType;

        Bucket(long startTime) {
            this.windowStart = startTime;
            this.adderForCounterType = new LongAdder[HystrixRollingNumberEvent.values().length];
            for (HystrixRollingNumberEvent type : HystrixRollingNumberEvent.values()) {
                if (!type.isCounter()) continue;
                this.adderForCounterType[type.ordinal()] = new LongAdder();
            }
            this.updaterForCounterType = new LongMaxUpdater[HystrixRollingNumberEvent.values().length];
            for (HystrixRollingNumberEvent type : HystrixRollingNumberEvent.values()) {
                if (!type.isMaxUpdater()) continue;
                this.updaterForCounterType[type.ordinal()] = new LongMaxUpdater();
                this.updaterForCounterType[type.ordinal()].update(0L);
            }
        }

        long get(HystrixRollingNumberEvent type) {
            if (type.isCounter()) {
                return this.adderForCounterType[type.ordinal()].sum();
            }
            if (type.isMaxUpdater()) {
                return this.updaterForCounterType[type.ordinal()].max();
            }
            throw new IllegalStateException("Unknown type of event: " + type.name());
        }

        LongAdder getAdder(HystrixRollingNumberEvent type) {
            if (!type.isCounter()) {
                throw new IllegalStateException("Type is not a Counter: " + type.name());
            }
            return this.adderForCounterType[type.ordinal()];
        }

        LongMaxUpdater getMaxUpdater(HystrixRollingNumberEvent type) {
            if (!type.isMaxUpdater()) {
                throw new IllegalStateException("Type is not a MaxUpdater: " + type.name());
            }
            return this.updaterForCounterType[type.ordinal()];
        }
    }

    private static class ActualTime
    implements Time {
        private ActualTime() {
        }

        @Override
        public long getCurrentTimeInMillis() {
            return System.currentTimeMillis();
        }
    }

    private static interface Time {
        public long getCurrentTimeInMillis();
    }
}

