/**
 *  Copyright Terracotta, Inc.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.sf.ehcache.writer.writebehind;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheEntry;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.PersistenceConfiguration.Strategy;
import net.sf.ehcache.store.Store;
import net.sf.ehcache.store.TerracottaStore;
import net.sf.ehcache.writer.CacheWriter;
import net.sf.ehcache.writer.CacheWriterManager;
import org.terracotta.statistics.Statistic;

/**
 * Implements a {@code WriterManager} that writes elements to a queue first and in the background sends the to the {@code CacheWriter}.
 *
 * @author Geert Bevin
 * @version $Id: WriteBehindManager.java 7936 2013-08-08 20:36:30Z cschanck $
 */
public class WriteBehindManager implements CacheWriterManager {
    private final WriteBehind writeBehind;

    /**
     * Create using the given cache.
     *
     * @param cache cache
     */
    public WriteBehindManager(final Cache cache, final Store store) {
        if (cache.isTerracottaClustered()) {
            writeBehind = ((TerracottaStore)store).createWriteBehind();
        } else if (cache.getCacheConfiguration().getPersistenceConfiguration() != null
                && cache.getCacheConfiguration().getPersistenceConfiguration().getStrategy() == Strategy.LOCALRESTARTABLE) {
            writeBehind = cache.getCacheManager().getFeaturesManager().createWriteBehind(cache);
        } else {
            writeBehind = new WriteBehindQueueManager(cache.getCacheConfiguration());
        }
    }

    /**
     * {@inheritDoc}
     */
    public void init(Cache cache) throws CacheException {
        CacheWriter cacheWriter = cache.getRegisteredCacheWriter();
        if (null == cacheWriter) {
            throw new CacheException("No cache writer was registered for cache " + cache.getName());
        }

        if (cache.getCacheConfiguration().getCacheWriterConfiguration().getWriteCoalescing()) {
            writeBehind.setOperationsFilter(new CoalesceKeysFilter());
        }

        writeBehind.start(cacheWriter);
    }

    /**
     * {@inheritDoc}
     */
    public void put(Element element) throws CacheException {
        writeBehind.write(element);
    }

    /**
     * {@inheritDoc}
     */
    public void remove(CacheEntry entry) throws CacheException {
        writeBehind.delete(entry);
    }

    /**
     * {@inheritDoc}
     */
    public void dispose() {
        /* If WriteBehind is configured but no writer is registered, writeBehind will be null */
        if (writeBehind != null) {
            writeBehind.stop();
        }
    }

    /**
     * Gets the best estimate for items in the queue still awaiting processing.
     * Not including elements currently processed
     *
     * @return the amount of elements still awaiting processing.
     */
    @Statistic(name = "queue-length", tags = "write-behind")
    public long getQueueSize() {
        return writeBehind.getQueueSize();
    }
}
