/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.healthmanager.metrics.timeline;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.util.Preconditions;

/**
 * CombinedTimelineAggregator aggregates value by partial timeline interval and then aggregates value of sub timeline aggregator by timeline interval.
 */
public class CombinedTimelineAggregator extends TimelineAggregator {

	private TimelineAggregator totalAggregator;
	private TimelineAggregator partialTimelineAggregator;
	private long lastTimestamp = -1;

	public CombinedTimelineAggregator(long startTimestamp, TimelineAggType timelineAggType,  long interval, TimelineAggType partialTimelineAggType, long partialTimelieInterval) {
		super(startTimestamp, interval);
		Preconditions.checkArgument(interval > partialTimelieInterval && interval % partialTimelieInterval == 0, "timeline interval mod sub timeline interval should be 0.");
		totalAggregator = TimelineAggregator.createTimelineAggregator(timelineAggType, interval, startTimestamp);
		partialTimelineAggregator = TimelineAggregator.createTimelineAggregator(partialTimelineAggType, partialTimelieInterval, startTimestamp);
	}

	@Override
	public void updateTimestamp(long timestamp) {
		partialTimelineAggregator.updateTimestamp(timestamp);
		Tuple2<Long, Double> subValue = partialTimelineAggregator.getValue();
		if (subValue != null && subValue.f0 > lastTimestamp) {
			totalAggregator.addValue(subValue);
			lastTimestamp = subValue.f0;
		}
		totalAggregator.updateTimestamp(timestamp);
	}

	@Override
	public void addValue(Tuple2<Long, Double> value) {
		partialTimelineAggregator.addValue(value);
		Tuple2<Long, Double> subValue = partialTimelineAggregator.getValue();
		if (subValue != null && subValue.f0 > lastTimestamp) {
			totalAggregator.addValue(subValue);
			lastTimestamp = subValue.f0;
		}
		totalAggregator.updateTimestamp(value.f0);
	}

	@Override
	public Tuple2<Long, Double> getValue() {
		return totalAggregator.getValue();
	}
}
