/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.healthmanager.metrics.timeline;

import org.apache.flink.api.java.tuple.Tuple2;

/**
 * Calculate max value of the metric in given interval.
 */
public class DeltaMaxTimelineAggregator extends TimelineAggregator {

	private long nextIntervalKey = -1;
	private double max = Double.NEGATIVE_INFINITY;
	private long lastTimestamp = -1;
	private double lastValue = 0;

	private long sampleInterval;

	public DeltaMaxTimelineAggregator(long interval) {
		super(0, interval);
		sampleInterval = interval / DELTA_SAMPLE_COUNT;
	}

	public DeltaMaxTimelineAggregator(long startTime, long interval) {
		super(startTime, interval);
		sampleInterval = interval / DELTA_SAMPLE_COUNT;
	}

	@Override
	public void updateTimestamp(long timestamp) {
		long intervalKey = getIntervalKey(timestamp);
		if (nextIntervalKey < intervalKey) {
			if (max != Double.NEGATIVE_INFINITY) {
				currentTimestamp = nextIntervalKey;
				currentValue = max;
			}
			nextIntervalKey = intervalKey;
			max = Double.NEGATIVE_INFINITY;
		}
	}

	@Override
	public void addValue(Tuple2<Long, Double> value) {
		if (lastTimestamp >= value.f0) {
			// duplicated data, ignored.
			return;
		}

		long intervalKey = getIntervalKey(value.f0);
		if (lastTimestamp + sampleInterval <= value.f0) {
			double delta = value.f1 - lastValue;
			lastValue = value.f1;
			lastTimestamp = value.f0;

			if (nextIntervalKey == intervalKey && max < delta) {
				max = delta;
			} else if (nextIntervalKey < intervalKey) {
				if (max != Double.NEGATIVE_INFINITY) {
					currentTimestamp = nextIntervalKey;
					currentValue = max;
				}
				nextIntervalKey = intervalKey;
				max = delta;
			}
		} else if (nextIntervalKey < intervalKey) {
			if (max != Double.NEGATIVE_INFINITY) {
				currentTimestamp = nextIntervalKey;
				currentValue = max;
			}
			nextIntervalKey = intervalKey;
			max = Double.NEGATIVE_INFINITY;
		}
	}
}
