/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.healthmanager.metrics.timeline;

import org.apache.flink.api.java.tuple.Tuple2;

import org.apache.commons.lang3.NotImplementedException;

/**
 * An TimelineAggregator will be aggregate value of a metric in a specified interval with different
 * aggregation function.
 */
public abstract class TimelineAggregator {

	protected long currentTimestamp = -1;
	protected double currentValue = Double.NEGATIVE_INFINITY;

	private long startTimestamp;
	private long interval;

	public static final int DELTA_SAMPLE_COUNT = 10;

	public TimelineAggregator(long startTimestamp, long interval) {
		this.startTimestamp = startTimestamp;
		this.interval = interval;
	}

	public long getIntervalKey(long ts) {
		return startTimestamp + (ts - startTimestamp) / interval * interval;
	}

	public abstract void updateTimestamp(long timestamp);

	/**
	 * Add a value to the timeline.
	 * @param value timestamp and value of a metric.
	 */
	public abstract void addValue(Tuple2<Long, Double> value);

	/**
	 * Get the value of the timeline after agg.
	 * @return timestamp and timeline.
	 */
	public Tuple2<Long, Double> getValue() {
		if (currentTimestamp == -1) {
			return null;
		}
		return Tuple2.of(currentTimestamp, currentValue);
	}

	/**
	 * Get the partial value of the timeline after agg. partial value means the value of latest window
	 * which may not be completed currently.
	 *
	 * @return
	 */
	public Tuple2<Long, Double> getPartialValue() {
		return null;
	}

	/**
	 * Create a timeline Aggregator.
	 * @param timelineAggType agg type of the aggregator
	 * @param interval        timeline intervale of the aggregator
	 * @return timeline aggregator with given agg type and interval.
	 */
	public static TimelineAggregator createTimelineAggregator(TimelineAggType timelineAggType, long interval) {
		return createTimelineAggregator(timelineAggType, interval, 0);
	}

	/**
	 * Create a timeline Aggregator.
	 * @param timelineAggType agg type of the aggregator
	 * @param interval        timeline intervale of the aggregator
	 * @return timeline aggregator with given agg type and interval.
	 */
	public static TimelineAggregator createTimelineAggregator(TimelineAggType timelineAggType, long interval, long startTime) {
		if (interval == 1L) {
			return new LatestValueAggregator(interval);
		}
		switch (timelineAggType) {
			case AVG:
				return new AvgTimelineAggregator(startTime, interval);
			case MAX:
				return new MaxTimelineAggregator(startTime, interval);
			case MIN:
				return new MinTimelineAggregator(startTime, interval);
			case RATE:
				return new RateTimelineAggregator(startTime, interval);
			case RANGE:
				return new RangeTimelineAggregator(startTime, interval);
			case LATEST:
				return new LatestTimelineAggregator(startTime, interval);
			case EARLIEST:
				return new EarliestTimelineAggregator(startTime, interval);
			case DELTA_AVG:
				return new DeltaAvgTimelineAggregator(startTime, interval);
			case DELTA_MIN:
				return new DeltaMinTimelineAggregator(startTime, interval);
			case DELTA_MAX:
				return new DeltaMaxTimelineAggregator(startTime, interval);
			default:
				throw new NotImplementedException("Agg type:" + timelineAggType);
		}
	}
}
