/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.healthmanager.metrics.timeline;

import org.apache.flink.annotation.VisibleForTesting;

/**
 * Interface to define a timeline aggregator.
 */
public interface TimelineAggregatorDescriptor {

	TimelineAggregator createTimelineAggregator();

	/**
	 * Descriptor for simple timeline agg.
	 */
	class SimpleTimelineAggregatorDescriptor implements TimelineAggregatorDescriptor {

		private TimelineAggType aggType;
		private long interval;
		private long startTime;

		@VisibleForTesting
		public SimpleTimelineAggregatorDescriptor(TimelineAggType aggType, long interval) {
			this.aggType = aggType;
			this.interval = interval;
			this.startTime = 0;
		}

		public SimpleTimelineAggregatorDescriptor(TimelineAggType aggType, long interval, long startTime) {
			this.aggType = aggType;
			this.interval = interval;
			this.startTime = startTime;
		}

		@Override
		public TimelineAggregator createTimelineAggregator() {
			return TimelineAggregator.createTimelineAggregator(aggType, interval, startTime);
		}
	}

	/**
	 * Descriptor for combined timeline agg.
	 */
	class CombinedTimelineAggregatorDescriptor implements TimelineAggregatorDescriptor {

		private TimelineAggType aggType;
		private long interval;
		private TimelineAggType partialAggType;
		private long partialInterval;
		private long startTime;

		@VisibleForTesting
		public CombinedTimelineAggregatorDescriptor(TimelineAggType aggType, long interval, TimelineAggType partialAggType, long partialInterval) {
			this.aggType = aggType;
			this.interval = interval;
			this.partialAggType = partialAggType;
			this.partialInterval = partialInterval;
			this.startTime = 0;
		}

		public CombinedTimelineAggregatorDescriptor(TimelineAggType aggType, long interval, TimelineAggType partialAggType, long partialInterval, long startTime) {
			this.aggType = aggType;
			this.interval = interval;
			this.partialAggType = partialAggType;
			this.partialInterval = partialInterval;
			this.startTime = startTime;
		}

		@Override
		public TimelineAggregator createTimelineAggregator() {
			return new CombinedTimelineAggregator(startTime, aggType, interval, partialAggType, partialInterval);
		}
	}

}
