/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.io.disk.iomanager;

import org.apache.flink.annotation.VisibleForTesting;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * The IORequest Scheduler which schedules the handling of IORequests.
 *
 * @param <E> The type of request.
 */
public class IORequestScheduler<E extends IORequest> implements Closeable {
	private static final Logger LOG = LoggerFactory.getLogger(IORequestScheduler.class);

	private volatile boolean closed = false;
	private final ConcurrentHashMap<FileIOChannel.ID, RequestQueue<E>> requestQueues = new ConcurrentHashMap<>();

	public RequestQueue<E> nextRequestQueue() throws InterruptedException {
		if (requestQueues.isEmpty()) {
			synchronized (this) {
				// double check
				if (requestQueues.isEmpty()) {
					wait();
				}
			}
		}

		long maxNumRequests = -1;
		FileIOChannel.ID channelID = null;
		for (Map.Entry<FileIOChannel.ID, RequestQueue<E>> entry : requestQueues.entrySet()) {
			if (entry.getValue().size() > maxNumRequests) {
				maxNumRequests = entry.getValue().size();
				channelID = entry.getKey();
			}
		}
		if (maxNumRequests > 0) {
			return requestQueues.get(channelID);
		}
		return null;
	}

	public void requestQueueProcessed(RequestQueue<E> requestQueue) {
		if (requestQueue.isEmpty()) {
			synchronized (this) {
				// double check
				if (requestQueue.isEmpty()) {
					requestQueues.remove(requestQueue.getFileChannelID());
					requestQueue.close();
				}
			}
		}
	}

	@VisibleForTesting
	public RequestQueue<E> getRequestQueue(FileIOChannel.ID channelID) {
		synchronized (this) {
			return requestQueues.get(channelID);
		}
	}

	public void addRequest(FileIOChannel.ID channelID, E request) {
		synchronized (this) {
			boolean isEmpty = requestQueues.isEmpty();
			RequestQueue<E> requestQueue = requestQueues.
				computeIfAbsent(channelID, key -> new RequestQueue<>(key));
			requestQueue.add(request);
			if (isEmpty) {
				notifyAll();
			}
		}
	}

	public void close() {
		synchronized (this) {
			for (RequestQueue<E> requestQueue : requestQueues.values()) {
				requestQueue.close();
			}
			closed = true;
		}
	}

	public boolean isClosed() {
		return closed;
	}

	public void shutDown(IOException ioex) {
		synchronized (this) {
			for (RequestQueue<E> requestQueue : requestQueues.values()) {
				while (!requestQueue.isEmpty()) {
					E request = requestQueue.poll();
					if (request != null) {
						try {
							request.requestDone(ioex);
						} catch (Throwable t) {
							IOManagerAsync.LOG.error("The handler of the request complete callback threw an exception"
								+ (t.getMessage() == null ? "." : ": " + t.getMessage()), t);
						}
					}
				}
			}
			requestQueues.clear();
		}
	}
}
