/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.	See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.	The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.	You may obtain a copy of the License at
 *
 *		 http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.io.network.partition.external;

import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataInputViewStreamWrapper;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicReference;

import static
	org.apache.flink.runtime.io.network.partition.external.ExternalBlockResultPartitionMeta.ExternalSubpartitionMeta;
import static org.apache.flink.util.Preconditions.checkArgument;

/**
 * Holds PartitionIndices for result partition in the format of {@code HASH_PARTITION_FILE}.
 */
public class HashPartitionIndices implements PartitionIndices {

	/** File system of data and index file for this partition */
	private final FileSystem fileSystem;

	/** The root directory of the result partition. */
	private final String partitionDir;

	/** The path of result partition index file. */
	private final Path indexFilePath;

	/** The number of subpartitions. */
	private final int subpartitionNum;

	/** The array of length of subpartitions. */
	@VisibleForTesting
	protected final AtomicReference<long[]> lengthArrayRef = new AtomicReference<>();

	public HashPartitionIndices(FileSystem fileSystem, String partitionDir, int subpartitionNum) {
		this.fileSystem = fileSystem;
		this.partitionDir = partitionDir;
		this.indexFilePath = new Path(ExternalBlockShuffleUtils.generateIndexPath(partitionDir, 0));
		this.subpartitionNum = subpartitionNum;
	}

	public void initialize() throws IOException {
		this.lengthArrayRef.set(loadPartitionIndices());
	}

	public ExternalSubpartitionMeta getSubpartitionMeta(int subpartitionIndex) throws IOException {
		checkArgument(subpartitionIndex >= 0 && subpartitionIndex < subpartitionNum, "Invalid subpartition index.");
		long[] tmpLengthArray = lengthArrayRef.get();
		// lengthArrayRef can be set null to make it GCable.
		if (tmpLengthArray == null) {
			tmpLengthArray = loadPartitionIndices();
			lengthArrayRef.set(tmpLengthArray);
		}
		return new ExternalSubpartitionMeta(
			new Path(ExternalBlockShuffleUtils.generateDataPath(partitionDir, subpartitionIndex)),
			0, tmpLengthArray[subpartitionIndex]);
	}

	public long shrinkMemoryFootprint() {
		long[] tmpLengthArray = lengthArrayRef.get();
		if (tmpLengthArray != null) {
			lengthArrayRef.lazySet(null);
			return 8L * subpartitionNum;
		} else {
			return 0;
		}
	}

	public long getShrinkableMemoryFootprint() {
		long[] tmpLengthArray = lengthArrayRef.get();
		if (tmpLengthArray != null) {
			return 8L * subpartitionNum;
		} else {
			return 0;
		}
	}

	private long[] loadPartitionIndices() throws IOException {
		// Checks whether index files exist
		if (!fileSystem.exists(indexFilePath)) {
			throw new IOException("Index file doesn't exist, file path: " + indexFilePath.getPath());
		}

		// Loads PartitionIndices from index files
		try (FSDataInputStream indexIn = fileSystem.open(indexFilePath)) {
			DataInputView indexView = new DataInputViewStreamWrapper(indexIn);

			// Gets the number of partitions indices in the index file.
			final int size = indexView.readInt();

			// Stores the overall length in the last element of the array.
			long[] tmpLengthArray = new long[subpartitionNum];
			int nextSubpartitionId = 0;
			for (int i = 0; i < size; i++) {
				int subpartitionId = indexView.readInt();

				// Fills offsets of non-existent partitions in case of noncontinuous partition IDs.
				while (nextSubpartitionId < subpartitionId) {
					tmpLengthArray[nextSubpartitionId++] = 0L;
				}

				if (nextSubpartitionId != subpartitionId) {
					throw new IOException("Got invalid partition id, expected nextSubpartitionId: "
						+ nextSubpartitionId + ", real subpartitionId: " + subpartitionId);
				}

				long startOffset = indexView.readLong();
				long lengthOfPartition = indexView.readLong();

				if (startOffset != 0L) {
					throw new IOException("Offset should be zero in HASH_PARTITION_FILE,"
						+ " partition id: " + subpartitionId + ", real startOffset: " + startOffset);
				}

				tmpLengthArray[nextSubpartitionId++] = lengthOfPartition;
			}

			while (nextSubpartitionId < subpartitionNum) {
				tmpLengthArray[nextSubpartitionId++] = 0;
			}
			assert nextSubpartitionId == subpartitionNum;

			return tmpLengthArray;
		} catch (IOException e) {
			throw new IOException("Cannot read index file, file path: " + indexFilePath.getPath(), e);
		}
	}
}
