/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.io.network.partition.external;

import org.apache.flink.annotation.VisibleForTesting;

import java.util.Comparator;
import java.util.PriorityQueue;

/**
 * An implementation of {@link ExternalBlockSubpartitionViewScheduler} which schedules
 * {@link ExternalBlockSubpartitionView}s according to the location of blocks to read on disks.
 */
public class LocalityBasedSubpartitionViewScheduler implements ExternalBlockSubpartitionViewScheduler {
	@VisibleForTesting
	protected final PriorityQueue<ExternalBlockSubpartitionView> subpartitionViewQueue;

	public LocalityBasedSubpartitionViewScheduler() {
		this.subpartitionViewQueue = new PriorityQueue<>(new LocalityBasedSubpartitionViewComparator());
	}

	@Override
	public void addToSchedule(ExternalBlockSubpartitionView subpartitionView) {
		subpartitionViewQueue.offer(subpartitionView);
	}

	@Override
	public ExternalBlockSubpartitionView schedule() {
		return subpartitionViewQueue.poll();
	}

	/**
	 * Compares subpartition views based on their relative locality.
	 */
	private static class LocalityBasedSubpartitionViewComparator implements Comparator<ExternalBlockSubpartitionView> {
		@Override
		public int compare(ExternalBlockSubpartitionView o1, ExternalBlockSubpartitionView o2) {
			if (!o1.getResultPartitionDir().equals(o2.getResultPartitionDir())) {
				return o1.getResultPartitionDir().compareTo(o2.getResultPartitionDir());
			}
			return o1.getSubpartitionIndex() - o2.getSubpartitionIndex();
		}
	}
}
