/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.resourcemanager.resultpartitionmaster;

import org.apache.flink.api.common.JobID;

/**
 * The job registration status used to check the whether a result partition has
 * been kept without a corresponding job for a while. If so, the result partition
 * need to be released.
 */
public class RegisteredJobStatus {
	private JobID jobId;

	private boolean online;

	private long offlineTimestamp;

	public RegisteredJobStatus(JobID jobId) {
		this.jobId = jobId;
		this.online = true;
	}

	public void markOffline() {
		if (online) {
			online = false;
			offlineTimestamp = System.currentTimeMillis();
		}
	}

	public void markOnline() {
		if (!online) {
			online = true;
			offlineTimestamp = 0;
		}
	}

	public boolean isTimeout(long timeoutInMills) {
		if (!online) {
			long now = System.currentTimeMillis();

			return now - offlineTimestamp > timeoutInMills;
		}

		return false;
	}

	public JobID getJobId() {
		return jobId;
	}
}
