/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.resourcemanager.slotmanager;

import org.apache.flink.api.common.time.Time;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.clusterframework.types.SlotID;
import org.apache.flink.runtime.clusterframework.types.TaskManagerSlot;
import org.apache.flink.runtime.concurrent.ScheduledExecutor;
import org.apache.flink.runtime.resourcemanager.SlotRequest;
import org.apache.flink.runtime.resourcemanager.placementconstraint.SlotTag;
import org.apache.flink.util.Preconditions;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * A slot manager that answers request with the slot which exactly the same resource with the request.
 */
public class StrictlyMatchingSlotManager extends SlotManager {

	private Map<ResourceID, Set<SlotTag>> tmTags = new HashMap<>();

	public StrictlyMatchingSlotManager(
		ScheduledExecutor scheduledExecutor,
		Time taskManagerRequestTimeout,
		Time slotRequestTimeout,
		Time taskManagerTimeout,
		Time taskManagerCheckerInitialDelay) {
		this(scheduledExecutor, taskManagerRequestTimeout, slotRequestTimeout, taskManagerTimeout, Time.milliseconds(0), taskManagerCheckerInitialDelay);
	}

	public StrictlyMatchingSlotManager(
		ScheduledExecutor scheduledExecutor,
		Time taskManagerRequestTimeout,
		Time slotRequestTimeout,
		Time taskManagerTimeout,
		Time taskManagerFastTimeout,
		Time taskManagerCheckerInitialDelay) {
		super(scheduledExecutor, taskManagerRequestTimeout, slotRequestTimeout, taskManagerTimeout, taskManagerFastTimeout, taskManagerCheckerInitialDelay);
	}

	@Override
	protected TaskManagerSlot findMatchingSlot(SlotRequest slotRequest) {
		Map<ResourceID, Integer> numTmFreeSlots = new HashMap<>();
		for (SlotID slotID : freeSlots.keySet()) {
			ResourceID resourceID = slotID.getResourceID();
			numTmFreeSlots.put(resourceID, numTmFreeSlots.getOrDefault(resourceID, 0) + 1);
		}

		Iterator<Map.Entry<SlotID, TaskManagerSlot>> iterator = freeSlots.entrySet().stream()
			.sorted((entry1, entry2) -> {
				int comp = numTmFreeSlots.get(entry1.getKey().getResourceID()) - numTmFreeSlots.get(entry2.getKey().getResourceID());
				if (comp == 0) {
					comp = entry1.getKey().getResourceID().hashCode() - entry2.getKey().getResourceID().hashCode();
				}
				return comp;
			})
			.iterator();
		while (iterator.hasNext()) {
			TaskManagerSlot taskManagerSlot = iterator.next().getValue();

			// sanity check
			Preconditions.checkState(taskManagerSlot.getState() == TaskManagerSlot.State.FREE,
				String.format("Slot %s is in state %s", taskManagerSlot.getSlotId(), taskManagerSlot.getState()));

			Set<SlotTag> tags = tmTags.get(taskManagerSlot.getSlotId().getResourceID());
			if (taskManagerSlot.getResourceProfile().equals(slotRequest.getResourceProfile()) &&
				(tags == null || tags.equals(slotRequest.getTags())) &&
				placementConstraintManager.check(
					slotRequest.getJobId(),
					allocationIdTags.get(slotRequest.getAllocationId()),
					getTaskExecutorSlotTags(taskManagerSlot.getSlotId()))) {
				freeSlots.remove(taskManagerSlot.getSlotId());
				return taskManagerSlot;
			}
		}
		return null;
	}

	@Override
	protected PendingSlotRequest findMatchingRequest(TaskManagerSlot taskManagerSlot) {
		for (PendingSlotRequest pendingSlotRequest : pendingSlotRequests.values()) {
			Set<SlotTag> tags = tmTags.get(taskManagerSlot.getSlotId().getResourceID());
			if (taskManagerSlot.getResourceProfile().equals(pendingSlotRequest.getResourceProfile()) &&
				!pendingSlotRequest.isAssigned() &&
				(tags == null || tags.equals(new HashSet<>(pendingSlotRequest.getTags()))) &&
				placementConstraintManager.check(
					pendingSlotRequest.getJobId(),
					allocationIdTags.get(pendingSlotRequest.getAllocationId()),
					getTaskExecutorSlotTags(taskManagerSlot.getSlotId()))) {
				return pendingSlotRequest;
			}
		}
		return null;
	}

	public void setTmTags(ResourceID resourceId, Set<SlotTag> tags) {
		tmTags.put(resourceId, tags);
	}
}
