/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine;

import java.util.Map;

/**
 * GeminiKVRegion.
 * K is type of key.
 * V is type of Map.
 */
public interface AbstractGeminiKMap<K, MK, MV, M extends Map<MK, MV>> extends GeminiKV<K, M> {
	/**
	 * Returns true if there exists a mapping for the mapping key under the
	 * given key.
	 *
	 * @param key The key of the mapping
	 * @return True if there exists a mapping for the mapping key under the
	 * given key.
	 */
	boolean contains(K key, MK mapKey);

	/**
	 * Returns the current value associated with the mapping key under the
	 * given key.
	 *
	 * @param key    The key under which the mapping is to be retrieved.
	 * @param mapKey The mapping key whose value is to be retrieved.
	 * @return The value associated with the mapping key under the given key.
	 */
	MV get(K key, MK mapKey);

	/**
	 * Returns the current value associated with the mapping key under the given
	 * key, or {@code defaultMapValue} if the mapping key does not exist under
	 * the given key.
	 *
	 * @param key             The key under which the mapping is to be retrieved.
	 * @param mapKey          The mapping key whose value is to be retrieved.
	 * @param defaultMapValue The default value associated with the mapping key
	 *                        under the given key.
	 * @return The value associated with the mapping key under the given key, or
	 * {@code defaultMapValue} if the mapping key does not exist under
	 * the given key.
	 */
	MV getOrDefault(K key, MK mapKey, MV defaultMapValue);

	/**
	 * Associates a new value with the mapping key under the given key. If
	 * the mapping already exists under the given key, the mapping's value will
	 * be replaced with the given value.
	 *
	 * @param key      The key under which the mapping is to be added.
	 * @param mapKey   The mapping key with which the given value is to be
	 *                 associated.
	 * @param mapValue The value to be associated with the mapping key.
	 */
	void add(K key, MK mapKey, MV mapValue);

	/**
	 * Adds all the mappings in the given map into the map under the given
	 * key. The addition of the mappings is atomic, i.e., exceptions will be
	 * thrown if some of them fail to be added into the state.
	 *
	 * @param key      The key under which the mappings are to be added.
	 * @param mappings The mappings to be added into the state
	 */
	void addAll(K key, Map<? extends MK, ? extends MV> mappings);

	/**
	 * Removes the mapping with the mapping key from the map under the given
	 * key, if it is present.
	 *
	 * @param key    The key under which the mapping is to be removed.
	 * @param mapKey The mapping key whose mapping is to be removed.
	 */
	void remove(K key, MK mapKey);
}
