/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine;

import org.apache.flink.runtime.state.gemini.engine.fs.FileID;
import org.apache.flink.runtime.state.gemini.engine.fs.FileManager;
import org.apache.flink.runtime.state.gemini.engine.fs.FileWriter;
import org.apache.flink.runtime.state.gemini.engine.fs.GeminiOutputStream;

import java.io.IOException;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * An implementation of {@link org.apache.flink.runtime.state.gemini.engine.fs.FileWriter}
 * that can tolerant the underlying file system throw exception up to specified times.
 */
public class FTFileWriter extends FileWriter {

	private final FileManager fileManager;

	private final FileID fileID;

	private final String filePath;

	private int continuesFailCount = 0;

	private final int failCountThreshold;

	private final ExceptionStat exceptionStat;

	private final Object lock = new Object();

	private volatile boolean closed;

	public FTFileWriter(
		GeminiOutputStream geminiOutputStream,
		FileManager fileManager,
		FileID fileID,
		String filePath,
		int failCountThread,
		ExceptionStat exceptionStat) {

		super(geminiOutputStream);
		this.fileManager = checkNotNull(fileManager);
		this.fileID = checkNotNull(fileID);
		this.filePath = checkNotNull(filePath);
		this.failCountThreshold = failCountThread;
		this.exceptionStat = exceptionStat;
	}

	@Override
	public long getAddress() throws IOException {
		return fileManager.getAddress(fileID, getPos());
	}

	@Override
	public long getSize() throws IOException {
		return getPos();
	}

	@Override
	public FileID getFileID() {
		return fileID;
	}

	@Override
	public String getFilePath() {
		return filePath;
	}

	@Override
	public void increasFailCount() {
		continuesFailCount += 1;
		exceptionStat.addTotalWriterException(1);
	}

	@Override
	public void resetFailCount() {
		this.continuesFailCount = 0;
	}

	@Override
	public boolean isValid() {
		return continuesFailCount < failCountThreshold;
	}

	@Override
	public void close() throws IOException {
		if (!closed) {
			synchronized (lock) {
				if (closed) {
					return;
				}
				closed = true;
				super.close();
			}
		}
	}

	@Override
	public void sync() throws IOException {
		if (!closed) {
			synchronized (lock) {
				if (closed) {
					return;
				}
				super.sync();
			}
		}
	}

	@Override
	public String toString() {
		return "FileWriterImpl: {" +
			"fileID=" + fileID +
			", filePath=" + filePath +
			"}";
	}
}

