/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine;

import org.apache.flink.runtime.state.gemini.engine.memstore.GSValueMap;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValueMapEntry;

/**
 * GInternalKSortedMap.
 * K is type of key.
 * MK is type of Key of Sorted Map.
 * MV is type of value of Sorted Map.
 * TODO LIMITATION: now if do remove or add opertor to the map instance returned by head/tail/sub operator,
 * it don't affect the origin map.
 */
public interface GInternalKSortedMap<K, MK, MV> {
	/**
	 * Returns the entry with the smallest map key under the given key.
	 *
	 * @param key The key under which the retrieved entry locates.
	 * @return The entry with the smallest map key under the given key.
	 */
	GSValueMapEntry<MK, MV> firstEntry(K key);

	/**
	 * Returns the entry with the largest map key under the given key.
	 *
	 * @param key The key under which the retrieved entry locates.
	 * @return The entry with the largest map key under the given key.
	 */
	GSValueMapEntry<MK, MV> lastEntry(K key);

	/**
	 * Returns an iterator over all the mappings under the given key where the
	 * mapping's key is strictly less than {@code endMapKey}. The iterator is
	 * backed by the state, so changes to the iterator are reflected in the
	 * state, and vice-versa.
	 *
	 * @param key       The key whose mappings are to be iterated.
	 * @param endMapKey The high endpoint (exclusive) of the map keys in the
	 *                  mappings to be iterated.
	 * @return A GSValueMap over all the mappings under the given key where the
	 * mapping's key is equal to or greater than {@code endMapKey}.
	 */
	GSValueMap<MK, MV> head(K key, MK endMapKey);

	/**
	 * Returns an iterator over all the mappings under the given key where the
	 * mapping's key is equal to or greater than {@code startMapKey}. The
	 * iterator is backed by the state, so changes to the iterator are reflected
	 * in the state, and vice-versa.
	 *
	 * @param key         The key whose mappings are to be iterated.
	 * @param startMapKey The low endpoint (inclusive) of the map keys in the
	 *                    mappings to be iterated.
	 * @return A GSValueMap over all the mappings under the given key where the
	 * mapping's key is equal to or greater than {@code startMapKey}.
	 */
	GSValueMap<MK, MV> tail(K key, MK startMapKey);

	/**
	 * Returns an iterator over all the mappings under the given key where the
	 * mapping's key locates in the given range. The iterator is backed by the
	 * state, so changes to the iterator are reflected in the state, and
	 * vice-versa.
	 *
	 * @param key         The key whose mappings are to be iterated.
	 * @param startMapKey The low endpoint (inclusive) of the map keys in the
	 *                    mappings to be iterated.
	 * @param endMapKey   The high endpoint (exclusive) of the map keys in the
	 *                    mappings to be iterated.
	 * @return A GSValueMap over all the mappings under the given key where the
	 * mapping's key locates in the given range.
	 */
	GSValueMap<MK, MV> subMap(K key, MK startMapKey, MK endMapKey);
}
