/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine;

import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;

/**
 * GRegionIDImpl.
 */
public class GRegionIDImpl implements GRegionID {
	private final int regionCode;
	private final int regionID;
	private final int indexID;

	/**
	 * indexID is from 1~255, 0 is for data. and from 1 to 7 is used for gemini self. from 8 will be used for customer in future.
	 *
	 * @param indexID  0 is for data; 1~255 is for index.
	 * @param regionID is same as Key Group.
	 */
	public GRegionIDImpl(int indexID, int regionID) {
		this.indexID = indexID;
		this.regionID = regionID;
		this.regionCode = indexID << G_INDEX_BIT | regionID;
		check();
	}

	private void check() {
		if (this.indexID > 0xFF || this.regionID > 0xFF_FF_FF) {
			throw new GeminiRuntimeException(String.format("error indexID %d or regionID %d, indexID must be less than 255, regionID less than 16777215.",
				this.indexID,
				this.regionID));
		}
	}

	public GRegionIDImpl(int regionCode) {
		this.regionCode = regionCode;
		this.regionID = regionCode & G_REGION_ID_MARK;
		this.indexID = regionCode >>> G_INDEX_BIT;
	}

	@Override
	public int getRegionCode() {
		return regionCode;
	}

	@Override
	public int getId() {
		return regionID;
	}

	@Override
	public int getIndexID() {
		return indexID;
	}

	@Override
	public String toString() {
		return "_" + indexID + "_" + regionID;
	}
}
