/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine;

import java.util.Map;

/**
 * GeminiKV.
 * K is type of key.
 * V is type of value.
 */
public interface GeminiKV<K, V> {
	/**
	 * Associates the given value with the given key in the storage.
	 * If the storage previously contains a value for the given key,
	 * the old value will be replaced with the new value.
	 *
	 * @param key   The key with which the given value is to be associated.
	 * @param value The value to be associated with the given key.
	 */
	void put(K key, V value);

	/**
	 * Returns the value associated with the given key in the state.
	 * Null will be returned if this state contains no value for the key.
	 *
	 * @param key The key of the value to be retrieved.
	 * @return The value associated with the given key.
	 */
	V get(K key);

	/**
	 * Returns the value associated with the key in the state, or
	 * {@code defaultValue} if the key does not exist in the state.
	 *
	 * @param key          The key whose associated value is to be returned.
	 * @param defaultValue The default value of the key.
	 * @return The value to which the specified key is mapped, or
	 * {@code defaultValue} if the key does not exist in the state.
	 */
	V getOrDefault(K key, V defaultValue);

	/**
	 * Removes the value for the given key from the storage if it is present.
	 *
	 * @param key The key of the value to be removed.
	 */
	void remove(K key);

	/**
	 * Returns all the key/value map in the state.
	 *
	 * @return The key/value map in the state.
	 */
	Map<K, V> getAll();

	/**
	 * Put all the key/value of the state into map.
	 */
	void getAll(Map<K, V> results);

	/**
	 * Remove all the items in the state.
	 */
	void removeAll();

	/**
	 * Returns an iterable over all of the keys in the state.
	 *
	 * @return The iterable of all the keys in the state.
	 */
	Iterable<K> keys();

	/**
	 * Returns true if the state contains a pair whose key is equal to the given
	 * object.
	 *
	 * @param key The key whose presence in the state to be tested.
	 * @return True if the state contains the pair for the given key.
	 */
	boolean contains(K key);
}
