/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.dbms;

import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.gemini.engine.GConfiguration;
import org.apache.flink.runtime.state.gemini.engine.GeminiDB;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.filter.StateFilter;
import org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics;
import org.apache.flink.runtime.state.gemini.engine.metrics.ExceptionMetrics;
import org.apache.flink.runtime.state.gemini.engine.metrics.FileCacheMetrics;
import org.apache.flink.runtime.state.gemini.engine.metrics.FileCleanerMetrics;
import org.apache.flink.runtime.state.gemini.engine.metrics.GeminiMetrics;
import org.apache.flink.runtime.state.gemini.engine.metrics.HandlerMetrics;
import org.apache.flink.runtime.state.gemini.engine.page.compress.GCompressAlgorithm;
import org.apache.flink.runtime.state.gemini.time.TimeProvider;
import org.apache.flink.util.Preconditions;

import java.util.concurrent.atomic.AtomicLong;

/**
 * GContextImpl.
 */
public class GContextImpl implements GContext {

	private final GeminiDB geminiDB;

	private final GConfiguration gConfiguration;

	private final int startRegionId;

	private final int endRegionId;

	private volatile long curVersion;

	private Supervisor supervisor;

	private GeminiMetrics geminiMetrics;

	private CacheMetrics cacheMetrics;

	private HandlerMetrics handlerMetrics;

	private FileCacheMetrics fileCacheMetrics;

	private FileCleanerMetrics fileCleanerMetrics;

	private ExceptionMetrics exceptionMetrics;

	private TimeProvider timeProvider;

	private final long ttl;

	private StateFilter stateFilter;

	private MetricGroup dbMetricGroup;

	private MetricGroup fileManagerMetricGroup;

	/**
	 * version that is large than restoredVersion will be invalid.
	 */
	private volatile long restoredVersion;

	/**
	 * The number of all states's access.
	 */
	private AtomicLong accessNumber;

	//TODO config, we can set it in GTable, and now we don't support change compression.
	private final GCompressAlgorithm inPageCompressAlgorithm;
	private final GCompressAlgorithm flushWholePageCompressAlgorithm;

	public GContextImpl(
		GeminiDB geminiDB, int sRegionId, int eRegionID, GConfiguration gConfiguration) {
		this.geminiDB = Preconditions.checkNotNull(geminiDB);
		this.startRegionId = sRegionId;
		this.endRegionId = eRegionID;
		this.gConfiguration = gConfiguration;
		this.restoredVersion = -1;
		//TODO set to checkpointID
		this.curVersion = -1;
		this.ttl = gConfiguration.getTtl();
		this.accessNumber = new AtomicLong(0);
		inPageCompressAlgorithm = gConfiguration.getInPageGCompressAlgorithm();
		flushWholePageCompressAlgorithm = gConfiguration.getFlushWholePageGCompressAlgorithm();
	}

	@Override
	public GeminiDB getGeminiDB() {
		return geminiDB;
	}

	@Override
	public long getCurVersion() {
		return this.curVersion;
	}

	@Override
	public void increaseCurVersion() {
		this.curVersion++;
	}

	@Override
	public void setRestoredVersion(long version) {
		this.restoredVersion = version;
	}

	@Override
	public long getRestoredVersion() {
		return this.restoredVersion;
	}

	@Override
	public int getStartRegionId() {
		return startRegionId;
	}

	@Override
	public int getEndRegionId() {
		return endRegionId;
	}

	@Override
	public Supervisor getSupervisor() {
		return this.supervisor;
	}

	@Override
	public void setSupervisor(Supervisor supervisor) {
		this.supervisor = supervisor;
	}

	@Override
	public GConfiguration getGConfiguration() {
		return this.gConfiguration;
	}

	@Override
	public void setDBMetricGroup(MetricGroup metricGroup) {
		this.dbMetricGroup = metricGroup;
	}

	@Override
	public MetricGroup getDBMetricGroup() {
		return dbMetricGroup;
	}

	@Override
	public void setFileManagerMetricGroup(MetricGroup metricGroup) {
		fileManagerMetricGroup = metricGroup;
	}

	@Override
	public MetricGroup getFileManagerMetricGroup() {
		return fileManagerMetricGroup;
	}

	@Override
	public void setGeminiMetric(GeminiMetrics geminiMetric) {
		this.geminiMetrics = geminiMetric;
	}

	@Override
	public GeminiMetrics getGeminiMetric() {
		return geminiMetrics;
	}

	@Override
	public void setCacheMetric(CacheMetrics cacheMetrics) {
		this.cacheMetrics = cacheMetrics;
	}

	@Override
	public CacheMetrics getCacheMetric() {
		return cacheMetrics;
	}

	@Override
	public void setHandlerMetric(HandlerMetrics handlerMetric) {
		this.handlerMetrics = handlerMetric;
	}

	@Override
	public HandlerMetrics getHandlerMetric() {
		return handlerMetrics;
	}

	@Override
	public void setFileCacheMetrics(FileCacheMetrics fileCacheMetrics) {
		this.fileCacheMetrics = fileCacheMetrics;
	}

	@Override
	public FileCacheMetrics getFileCacheMetrics() {
		return fileCacheMetrics;
	}

	@Override
	public void setFileCleanerMetrics(FileCleanerMetrics fileCleanerMetrics) {
		this.fileCleanerMetrics = fileCleanerMetrics;
	}

	@Override
	public FileCleanerMetrics getFileCleanerMetrics() {
		return fileCleanerMetrics;
	}

	@Override
	public void setExceptionMetrics(ExceptionMetrics exceptionMetrics) {
		this.exceptionMetrics = exceptionMetrics;
	}

	@Override
	public ExceptionMetrics getExceptionMetrics() {
		return exceptionMetrics;
	}

	@Override
	public void setTimeProvider(TimeProvider timeProvider) {
		this.timeProvider = timeProvider;
	}

	@Override
	public TimeProvider getTimeProvider() {
		return timeProvider;
	}

	@Override
	public boolean hasTtl() {
		return ttl > 0;
	}

	@Override
	public boolean isExpired(long ts, long currentTime) {
		return hasTtl() && (ts + Math.min(Long.MAX_VALUE - ts, ttl) <= currentTime);
	}

	@Override
	public void setStateFilter(StateFilter stateFilter) {
		this.stateFilter = stateFilter;
	}

	@Override
	public StateFilter getStateFilter() {
		return this.stateFilter;
	}

	@Override
	public void incAccessNumber() {
		accessNumber.incrementAndGet();
	}

	@Override
	public long getAccessNumber() {
		return accessNumber.get();
	}

	@Override
	public long incrementAndGetAccessNumber() {
		return accessNumber.incrementAndGet();
	}

	@Override
	public long getMinSnapshotAccessNumber() {
		return supervisor.getSnapshotManager().getMinRunningSnapshotAccessNumber();
	}

	@Override
	public GCompressAlgorithm getInPageGCompressAlgorithm() {
		return inPageCompressAlgorithm;
	}

	@Override
	public GCompressAlgorithm getFlushWholePageGCompressAlgorithm() {
		return flushWholePageCompressAlgorithm;
	}

	@Override
	public void setDBInternalError(Throwable throwable) {
		geminiDB.setInternalError(throwable);
	}

	@Override
	public void checkDBStatus() {
		GeminiDB.Status status = geminiDB.getStatus();
		if (status != GeminiDB.Status.OPENED) {
			switch (status) {
				case CLOSING:
					throw new GeminiRuntimeException("Gemini is being closed");
				case CLOSED:
					throw new GeminiRuntimeException("Gemini has been closed");
				case INITIALIZE:
					throw new GeminiRuntimeException("Gemini has not been opened");
				case INTERNAL_ERROR:
					throw new GeminiRuntimeException("Internal error occurred, " + geminiDB.getInternalError());
			}
		}
	}

	@Override
	public GeminiDB.Status getDBStatus() {
		return geminiDB.getStatus();
	}

	@Override
	public boolean isDBNormal() {
		return geminiDB.getStatus() == GeminiDB.Status.OPENED;
	}
}
