/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.filecache;

import org.apache.flink.runtime.state.gemini.engine.GRegionContext;
import org.apache.flink.runtime.state.gemini.engine.page.DataPage;
import org.apache.flink.runtime.state.gemini.engine.page.PageAddress;
import org.apache.flink.util.Preconditions;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import javax.annotation.Nullable;

import java.io.Closeable;
import java.io.IOException;
import java.util.function.BiConsumer;

/**
 * Cache based on file. Generally, the page will store the data in the cache first,
 * and it depends on the implementation when to flush the page to the destination
 * storage. {@link #flushPage} can be used to flush the page to the destination storage
 * directly. It depends on the implementation whether the methods are blocked on IO.
 */
public abstract class FileCache implements Closeable {

	public static final Long INFINITE_CAPACITY = Long.MAX_VALUE;

	/**
	 * Capacity of the cache.
	 */
	protected final long capacity;

	protected final FileCacheStat fileCacheStat;

	public FileCache(long capacity, FileCacheStat fileCacheStat) {
		this.capacity = capacity;
		this.fileCacheStat = Preconditions.checkNotNull(fileCacheStat);
	}

	/**
	 * Return the capacity of the cache.
	 */
	public long capacity() {
		return capacity;
	}

	/**
	 * Whether the page is cached.
	 */
	public abstract boolean isCached(PageAddress pageAddress);

	/**
	 * Add the page to cache. Must guarantee the page is in memory.
	 * It depends on the implementation whether this method is sync
	 * or async. It can use {@param eventExecutor} to be async.
	 * We will run the {@param callBack} finally, and deliver a boolean
	 * argument which indicates whether the page has been successfully
	 * added.
	 *
	 * @param pageAddress address of page to add.
	 * @param gRegionContext context of region this page belongs to.
	 * @param eventExecutor executor used to be async.
	 * @param callBack code to execute after page is added to cache.
	 */
	public abstract void addPage(
		PageAddress pageAddress,
		GRegionContext gRegionContext,
		EventExecutor eventExecutor,
		@Nullable BiConsumer<Boolean, Throwable> callBack);

	/**
	 * Get the page.
	 */
	public abstract DataPage getPage(
		PageAddress pageAddress,
		GRegionContext gRegionContext,
		EventExecutor eventExecutor);

	/**
	 * Discard the page from the cache and destination storage.
	 * It depends on the implementation whether this method is
	 * sync or async. It can use {@param eventExecutor} to be async.
	 *
	 * @param pageAddress address of page to discard.
	 * @param gRegionContext context of region this page belongs to.
	 * @param eventExecutor executor used to be async.
	 */
	public abstract void discardPage(
		PageAddress pageAddress,
		GRegionContext gRegionContext,
		EventExecutor eventExecutor);

	/**
	 * Flush the page to the destination storage without caching.
	 * It depends on the implementation whether this method is sync or async.
	 * It can use {@param eventExecutor} to be async. If {@param force} is true,
	 * the page will always be flushed no matter whether it has been flushed,
	 * and replace the old address. We will run the {@param callBack} finally,
	 * and deliver a boolean argument which indicates whether the page has been
	 * successfully added.
	 *
	 * @param pageAddress address of page to flush.
	 * @param gRegionContext context of region this page belongs to.
	 * @param eventExecutor executor used to be async.
	 * @param force whether to force to flush.
	 * @param callBack code to execute after page is flushed.
	 */
	public abstract void flushPage(
		PageAddress pageAddress,
		GRegionContext gRegionContext,
		EventExecutor eventExecutor,
		boolean force,
		@Nullable BiConsumer<Boolean, Throwable> callBack);

	/**
	 * TODO this is a hack method to sync snapshot data. We will improve it later.
	 */
	public abstract void sync(EventExecutor eventExecutor) throws IOException;

	public FileCacheStat getFileCacheStat() {
		return fileCacheStat;
	}
}
