/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.filecache;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.concurrent.atomic.AtomicLong;

/**
 * Statistics for file cache.
 */
public class FileCacheStat {

	public AtomicLong hitCount = new AtomicLong(0);

	public AtomicLong hitSize = new AtomicLong(0);

	public AtomicLong missCount = new AtomicLong(0);

	public AtomicLong missSize = new AtomicLong(0);

	public AtomicLong totalLocalWriteCount = new AtomicLong(0);

	public AtomicLong totalLocalWriteTime = new AtomicLong(0);

	public AtomicLong totalLocalWriteSize = new AtomicLong(0);

	public AtomicLong totalLocalOriDataSize = new AtomicLong(0);

	public AtomicLong totalLocalReadCount = new AtomicLong(0);

	public AtomicLong totalLocalReadTime = new AtomicLong(0);

	public AtomicLong totalLocalReadSize = new AtomicLong(0);

	public AtomicLong totalDFSWriteCount = new AtomicLong(0);

	public AtomicLong totalDFSWriteTime = new AtomicLong(0);

	public AtomicLong totalDFSWriteSize = new AtomicLong(0);

	public AtomicLong totalDFSReadCount = new AtomicLong(0);

	public AtomicLong totalDFSReadTime = new AtomicLong(0);

	public AtomicLong totalDFSReadSize = new AtomicLong(0);

	public void addHitSize(long size) {
		hitCount.addAndGet(1);
		hitSize.addAndGet(size);
	}

	public void addMissSize(long size) {
		missCount.addAndGet(1);
		missSize.addAndGet(size);
	}

	public void addLocalWrite(long writeSize, long oriDataSize, long writeTime) {
		totalLocalWriteCount.addAndGet(1);
		totalLocalWriteSize.addAndGet(writeSize);
		totalLocalOriDataSize.addAndGet(oriDataSize);
		totalLocalWriteTime.addAndGet(writeTime);
	}

	public void addLocalRead(long readSize, long readTime) {
		totalLocalReadCount.addAndGet(1);
		totalLocalReadSize.addAndGet(readSize);
		totalLocalReadTime.addAndGet(readTime);
	}

	public void addDFSWrite(long writeSize, long writeTime) {
		totalDFSWriteCount.addAndGet(1);
		totalDFSWriteSize.addAndGet(writeSize);
		totalDFSWriteTime.addAndGet(writeTime);
	}

	public void addDFSRead(long readSize, long readTime) {
		totalDFSReadCount.addAndGet(1);
		totalDFSReadSize.addAndGet(readSize);
		totalDFSReadTime.addAndGet(readTime);
	}

	private long getAverage(AtomicLong a, AtomicLong b) {
		if (b.get() == 0) {
			return 0;
		}
		return a.get() / b.get();
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("hitCount", hitCount.get()).
			add("hitSize", hitSize.get()).
			add("missCount", missCount.get()).
			add("missSize", missSize.get()).
			add("averageLocalWriteSize", getAverage(totalLocalWriteSize, totalLocalWriteCount)).
			add("averageLocalWriteTime(ns)", getAverage(totalLocalWriteTime, totalLocalWriteCount)).
			add("averageLocalReadSize", getAverage(totalLocalReadSize, totalLocalReadCount)).
			add("averageLocalReadTime(ns)", getAverage(totalLocalReadTime, totalLocalReadCount)).
			add("averageDFSWriteSize", getAverage(totalDFSWriteSize, totalDFSWriteCount)).
			add("averageDFSWriteTime(ns)", getAverage(totalDFSWriteTime, totalDFSWriteCount)).
			add("averageDFSReadSize", getAverage(totalDFSReadSize, totalDFSReadCount)).
			add("averageDFSReadTime(ns)", getAverage(totalDFSReadTime, totalDFSReadCount)).
			toString();
	}
}
