/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import java.io.Closeable;
import java.util.Collection;

/**
 * Clean up discarded files in registered {@link FileManager}s and
 * files manually registered by {@link #registerFilesToClean(Collection)}.
 *
 * <p>File cleaner can work in two ways:
 * 1. File cleaner call {@link FileManager#getMarkedDeletionFiles()} to
 *    get a set of files to discarded. It's up to the file cleaner
 *    when to clean up files.
 * 2. File manager call {@link #triggerCleanup(FileManager)} to tell
 *    file cleaner that it has files to be discarded.
 */
public interface FileCleaner extends Closeable {

	void start();

	/**
	 * Register a file manager.
	 */
	void registerFileManager(FileManager fileManager);

	/**
	 * Unregister a file manager.
	 */
	void unregisterFileManager(FileManager fileManager);

	/**
	 * Trigger a cleanup for the file manager.
	 */
	void triggerCleanup(FileManager fileManager);

	/**
	 * Register files to clean.
	 *
	 * @param fileToClean The path of files to clean.
	 * @return <tt>true</tt> if this collection registered successfully.
	 */
	boolean registerFilesToClean(Collection<String> fileToClean);
}
