/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.util.Preconditions;

import java.util.Objects;

/**
 * Implementation of file id. We have three parts to compose the whole id: flag, sub-task-index, unique increased id in a circular range.
 * For total tasks number in range [1, 1024], we have distribution as: |1_bit|10_bit|21_bit|
 * FOr total tasks number in range [1025, 32768], we have distribution as: |2_bit|15_bit|15_bit|
 */
public class FileIDImpl implements FileID{
	private final FileIDGenerator.FileIdMode fileIdMode;

	private final int subTaskIndex;

	private final int uniqueId;

	private final int value;

	public FileIDImpl(int fileIDValue) {
		FileIDGenerator.FileIdMode fileIdMode = null;
		for (FileIDGenerator.FileIdMode idMode : FileIDGenerator.FileIdMode.values()) {
			if (fileIDValue >>> idMode.getFlagOffset() == idMode.getFlagValue()) {
				fileIdMode = idMode;
				break;
			}
		}
		if (fileIdMode == null) {
			throw new IllegalArgumentException("Fail to parse given fileIDValue " + fileIDValue
				+ " (binaryString: " + Integer.toBinaryString(fileIDValue)+ ") to expected file modes.");
		}
		this.fileIdMode = fileIdMode;
		this.subTaskIndex = (fileIDValue & fileIdMode.getSubtaskIndexMask()) >>> fileIdMode.getSubtaskIndexOffset();
		this.uniqueId = fileIDValue & fileIdMode.getUniqueIDMask();
		this.value = fileIDValue;
	}

	public FileIDImpl(FileIDGenerator.FileIdMode fileIdMode, int subTaskIndex, int uniqueId) {
		Preconditions.checkArgument(subTaskIndex < fileIdMode.getMaxAllowedTaskNum() && subTaskIndex >= 0,
			"Illegal of sub task index: " + subTaskIndex + "，less than zero or larger than the max allowed total task num " +
			fileIdMode.getMaxAllowedTaskNum() + " of fileID mode: " + fileIdMode);

		Preconditions.checkArgument(uniqueId <= fileIdMode.getMaxAllowedUniqueID(),
			"Illegal of uniqueId: " + uniqueId + ", larger than the max allowed uniqueId " +
				fileIdMode.getMaxAllowedUniqueID() + "  of fileId mode: " + fileIdMode);

		this.fileIdMode = fileIdMode;
		this.subTaskIndex = subTaskIndex;
		this.uniqueId = uniqueId;
		this.value = fileIdMode.getInitValue() | (subTaskIndex << fileIdMode.getSubtaskIndexOffset()) | uniqueId;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		FileIDImpl fileID = (FileIDImpl) o;
		return value == fileID.value;
	}

	@Override
	public int hashCode() {
		return Objects.hash(value);
	}

	@Override
	public int get() {
		return value;
	}

	@Override
	public int getUniqueID() {
		return uniqueId;
	}

	@Override
	public int getMaxAllowedUniqueID() {
		return fileIdMode.getMaxAllowedUniqueID();
	}

	public int getSubTaskIndex() {
		return subTaskIndex;
	}

	public FileIDGenerator.FileIdMode getFileIdMode() {
		return fileIdMode;
	}

	@Override
	public String toString() {
		return "FileID{" +
			"fileIdMode=" + fileIdMode +
			", subTaskIndex=" + subTaskIndex +
			", uniqueId=" + uniqueId +
			'}';
	}
}
