/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.gemini.engine.page.DataPageUtil;

import java.io.Closeable;
import java.util.Map;
import java.util.Set;

/**
 * FileManager.
 */
public interface FileManager extends Closeable {

	void start();

	String getFileManagerIdentifier();

	Path getBasePath();

	String getFilePath(FileID fileID);

	String getFilePath(int fileId);

	/**
	 * Get the {@link FileID} from the address.
	 */
	FileID getFileID(long address);

	/**
	 * Return an address according to the file id and file offset.
	 */
	long getAddress(FileID fileID, long offset);

	/**
	 * Get the file offset from the address.
	 */
	long getFileOffset(long address);

	/**
	 * Get the {@link FileReader} for the given address.
	 */
	FileReader getFileReader(long address);

	/**
	 * Create a new {@link FileWriter}.
	 */
	FileWriter createNewFileWriter();

	/**
	 * Close the file writer.
	 */
	void closeFileWriter(FileWriter fileWriter);

	/**
	 * Increment the DB reference for the file.
	 *
	 * @param address the address containing of the file information.
	 * @param dataSize the size of data this reference holds.
	 */
	void incDBReference(long address, long dataSize);

	/**
	 * Decrement the DB reference for the file.
	 *
	 * @param address the address containing of the file information.
	 * @param accessNumber the access number when this decrement happens.
	 * @param ts the timestamp when this decrement happens.
	 * @param dataSize the size of data this reference holds is.
	 */
	void decDBReference(long address, long accessNumber, long ts, long dataSize);

	/**
	 * Increment the snapshot reference for the file.
	 */
	void incSnapshotReference(FileID fileID);

	/**
	 * Decrement the snapshot reference for the file.
	 */
	void decSnapshotReference(FileID fileID);

	/**
	 * Return a set of marked deletion files, used for file cleanup.
	 */
	Set<String> getMarkedDeletionFiles();

	void restore(Map<Integer, FileMeta.RestoredFileMeta> restoredFileMapping);

	/**
	 * Get the file mapping according of the given set of file id.
	 */
	Map<FileID, String> getFileMapping(Set<FileID> fileIDs);

	/**
	 * Increase fileWriter error count.
	 */
	void increaseFileWriterErrorCount();

	/**
	 * Reset the fileWriter error count to fresh.
	 */
	void resetFileWriterErrorCount();

	/**
	 * @return True, if current file manager is valid, otherwise return false.
	 */
	boolean isValid();

	DataPageUtil getDataPageUtil();
}
