/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Statistics for file manager.
 */
public class FileManagerStat {

	public AtomicInteger maxUsedFile;

	public AtomicInteger numberUsedFile;

	public AtomicInteger numberWaitingDBDeletionFile;

	public AtomicInteger numberWaitingSnapshotDeletionFile;

	public AtomicInteger numberMarkDeletionFile;

	public AtomicInteger totalCreatedFile;

	public AtomicInteger totalFailCreateFile;

	public AtomicInteger totalDeletedFile;

	public AtomicLong totalDataSize;

	public FileManagerStat() {
		this.maxUsedFile = new AtomicInteger(0);
		this.numberUsedFile = new AtomicInteger(0);
		this.numberWaitingDBDeletionFile = new AtomicInteger(0);
		this.numberWaitingSnapshotDeletionFile = new AtomicInteger(0);
		this.numberMarkDeletionFile = new AtomicInteger(0);
		this.totalCreatedFile = new AtomicInteger(0);
		this.totalFailCreateFile = new AtomicInteger(0);
		this.totalDeletedFile = new AtomicInteger(0);
		this.totalDataSize = new AtomicLong(0);
	}

	public void setMaxUsedFile(int number) {
		int prev = maxUsedFile.get();
		int cur = Math.max(prev, number);
		maxUsedFile.compareAndSet(prev, cur);
	}

	public void setNumberUsedFile(int number) {
		numberUsedFile.set(number);
	}

	public void setNumberWaitingDBDeletionFile(int number) {
		numberWaitingDBDeletionFile.set(number);
	}

	public void setNumberWaitingSnapshotDeletionFile(int number) {
		numberWaitingSnapshotDeletionFile.set(number);
	}

	public void setNumberMarkDeletionFile(int number) {
		numberMarkDeletionFile.set(number);
	}

	public int addTotalCreatedFile(int delta) {
		return totalCreatedFile.addAndGet(delta);
	}

	public int addTotalFailCreateFile(int delta) {
		return totalFailCreateFile.addAndGet(delta);
	}

	public int addTotalDeletedFile(int delta) {
		return totalDeletedFile.addAndGet(delta);
	}

	public long addTotalDataSize(long dataSize) {
		return totalDataSize.addAndGet(dataSize);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("maxUsedFile", maxUsedFile.get()).
			add("numberUsedFile", numberUsedFile.get()).
			add("numberWaitingDBDeletionFile", numberWaitingDBDeletionFile.get()).
			add("numberWaitingSnapshotDeletionFile", numberWaitingSnapshotDeletionFile.get()).
			add("numberMarkDeletionFile", numberMarkDeletionFile.get()).
			add("totalCreatedFile", totalCreatedFile.get()).
			add("totalFailCreateFile", totalFailCreateFile.get()).
			add("totalDeletedFile", totalDeletedFile.get()).
			add("totalDataSize", totalDataSize.get()).
			toString();
	}
}
