/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import javax.annotation.Nullable;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Meta information of a file.
 */
public class FileMeta {

	/**
	 * Whether this file can be deleted when it's discarded. If this file is
	 * restored from a namespace which is different from the current namespace
	 * (eg, rescale or resume in Flink), we can not deleted it, and it should
	 * be managed by CheckpointCoordinator.
	 */
	private final boolean canDeleted;

	/**
	 * File path.
	 */
	private final String filePath;

	/**
	 * ID of the file.
	 */
	private final FileID fileId;

	/**
	 * Size of the file. It may be not update in real time.
	 */
	private long fileSize;

	/**
	 * Size of useful data.
	 */
	private AtomicLong dataSize;

	/**
	 * Count for the number of references by DB currently.
	 */
	private AtomicInteger dbReference;

	private AtomicInteger snapshotReference;

	/**
	 * Timestamp when this file is no more referenced by DB,
	 * that's dbReference becomes 0.
	 */
	private long discardTimeStamp;

	/**
	 * Access number when this file is no more referenced by DB,
	 * that's dbReference becomes 0.
	 */
	private long discardAccessNumber;

	/**
	 * A {@link FileWriter} to write data to the file. It can be null when
	 * this file becomes read-only or is no longer referenced by DB.
	 */
	@Nullable
	private FileWriter fileWriter;

	/**
	 * A {@link FileReader} to read data from the file. The reader itself
	 * may not be thread safe, and the caller should be responsible for this.
	 * The reader is lazily created, and can be null when the file meta is
	 * created or the file is no longer referenced by DB.
	 */
	@Nullable
	private FileReader fileReader;

	/**
	 * Used when creating a new file.
	 */
	public FileMeta(
		String filePath,
		FileID fileId
	) {
		this(filePath,
			fileId,
			0,
			0,
			0,
			0,
			true
		);
	}

	public FileMeta(
		String filePath,
		FileID fileId,
		long fileSize,
		long dataSize,
		int dbReference,
		int snapshotReference,
		boolean canDeleted
	) {
		this.filePath = filePath;
		this.fileId = fileId;
		this.fileSize = fileSize;
		this.dataSize = new AtomicLong(dataSize);
		this.dbReference = new AtomicInteger(dbReference);
		this.snapshotReference = new AtomicInteger(snapshotReference);
		this.canDeleted = canDeleted;
		this.discardAccessNumber = -1;
		this.discardTimeStamp = -1;
	}

	public String getFilePath() {
		return filePath;
	}

	public FileID getFileId() {
		return fileId;
	}

	public void setFileSize(long fileSize) {
		this.fileSize = fileSize;
	}

	public long getFileSize() {
		return fileSize;
	}

	public long addAndGetDataSize(long size) {
		return dataSize.addAndGet(size);
	}

	public int addAndGetDBReference(int delta) {
		return dbReference.addAndGet(delta);
	}

	public int addAndGetSnapshotReference(int delta) {
		return snapshotReference.addAndGet(delta);
	}

	public long getDiscardTimeStamp() {
		return discardTimeStamp;
	}

	public long getDiscardAccessNumber() {
		return discardAccessNumber;
	}

	public void updateDiscardAccessNumberAndTimestamp(long accessNumber, long timestamp) {
		synchronized (this) {
			this.discardAccessNumber = Math.max(accessNumber, discardAccessNumber);
			this.discardTimeStamp = Math.max(timestamp, discardTimeStamp);
		}
	}

	public void setFileWriter(FileWriter fileWriter) {
		this.fileWriter = fileWriter;
	}

	public FileWriter getFileWriter() {
		return fileWriter;
	}

	public void setFileReader(FileReader fileReader) {
		this.fileReader = fileReader;
	}

	public FileReader getFileReader() {
		return fileReader;
	}

	public boolean canDeleted() {
		return this.canDeleted;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (!(o instanceof FileMeta)) {
			return false;
		}

		FileMeta fileMeta = (FileMeta) o;
		return Objects.equals(this.fileId, fileMeta.fileId)
			&& Objects.equals(this.filePath, fileMeta.filePath)
			&& (canDeleted == fileMeta.canDeleted);
	}

	@Override
	public int hashCode() {
		int result = Objects.hashCode(fileId);
		result = 31 * result + Objects.hashCode(filePath);
		result = 31 * result + Objects.hashCode(canDeleted);
		return result;
	}

	@Override
	public String toString() {
		return "FileMeta{" +
			"filePath=" + filePath +
			", fileId=" + fileId +
			", canDeleted=" + canDeleted +
			", fileSize=" + fileSize +
			", dataSize=" + dataSize +
			", dataRef=" + dbReference +
			", snapshotRef=" + snapshotReference +
			", discardTimeStamp=" + discardTimeStamp +
			", discardAccessNumber=" + discardAccessNumber +
			'}';
	}

	/**
	 * Convenience for restore file.
	 */
	public static class RestoredFileMeta {

		public int id;

		public String filePath;

		public long fileSize;

		public long dataSize;

		public int dbReference;

		public int snapshotReference;

		public boolean canDeleted;

		public static RestoredFileMeta of (
			int id,
			String filePath,
			long fileSize,
			long dataSize,
			int dbReference,
			int snapshotReference,
			boolean canDeleted) {
			RestoredFileMeta restoredFileMeta = new RestoredFileMeta();
			restoredFileMeta.id = id;
			restoredFileMeta.filePath = filePath;
			restoredFileMeta.fileSize = fileSize;
			restoredFileMeta.dataSize = dataSize;
			restoredFileMeta.dbReference = dbReference;
			restoredFileMeta.snapshotReference = snapshotReference;
			restoredFileMeta.canDeleted = canDeleted;
			return restoredFileMeta;
		}
	}
}
