/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.util.Preconditions;

import java.io.IOException;

/**
 * An implementation of {@link FileWriter}.
 */
public class FileWriterImpl extends FileWriter {

	// File manager this writer belongs to.
	private final FileManager fileManager;

	// ID of the file this write belongs to.
	private final FileID fileID;

	private final String filePath;

	private final Object lock = new Object();

	private volatile boolean closed;

	public FileWriterImpl(
		GeminiOutputStream geminiOutputStream,
		FileManager fileManager,
		FileID fileID,
		String filePath
	) {
		super(geminiOutputStream);
		this.fileManager = Preconditions.checkNotNull(fileManager);
		this.fileID = Preconditions.checkNotNull(fileID);
		this.filePath = filePath;
		this.closed = false;
	}

	@Override
	public FileID getFileID() {
		return fileID;
	}

	@Override
	public String getFilePath() {
		return filePath;
	}

	@Override
	public long getAddress() throws IOException {
		return fileManager.getAddress(fileID, getPos());
	}

	@Override
	public long getSize() throws IOException {
		return getPos();
	}

	@Override
	public void close() throws IOException {
		if (!closed) {
			synchronized (lock) {
				if (closed) {
					return;
				}
				closed = true;
				super.close();
			}
		}
	}

	@Override
	public void sync() throws IOException {
		if (!closed) {
			synchronized (lock) {
				if (closed) {
					return;
				}
				super.sync();
			}
		}
	}

	@Override
	public boolean isValid() {
		// always valid.
		return true;
	}

	@Override
	public void increasFailCount() {
		// do nothing.
	}

	@Override
	public void resetFailCount() {
		// do nothing.
	}

	@Override
	public String toString() {
		return "FileWriterImpl: {" +
			"fileID=" + fileID +
			", filePath=" + filePath +
			"}";
	}
}
