/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.fs.hdfs.HadoopDataInputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

import java.io.IOException;
import java.nio.ByteBuffer;

/**
 * GeminiInputStream.
 */
public class GeminiInputStream extends FSDataInputStream {

	private static final Logger LOG = LoggerFactory.getLogger(GeminiInputStream.class);

	private static final int DEFAULT_BUFFER_SIZE = 4 * 1024;

	private final FileSystem fileSystem;

	private final Path filePath;

	private FSDataInputStream inputStream;

	public GeminiInputStream(
		Path filePath
	) throws IOException {
		this(filePath, DEFAULT_BUFFER_SIZE);
	}

	public GeminiInputStream(
		Path filePath,
		int bufferSize
	) throws IOException {
		this.filePath = filePath;
		this.fileSystem = FileSystem.getUnguardedFileSystem(filePath.toUri());

		// for local file system, opening a file with buffer size takes no effect.
		// wrap it with a buffered stream.
		if (!fileSystem.isDistributedFS()) {
//			FSDataInputStream stream = fileSystem.open(filePath);
//			this.inputStream = new GeminiBufferedInputStream(stream, bufferSize);
			this.inputStream = new GeminiLocalByteBufferableInputStream(fileSystem.getWorkingDirectory(), filePath);
		} else {
			this.inputStream = fileSystem.open(filePath, bufferSize);
		}
	}

	@Override
	public long getPos() throws IOException {
		return inputStream.getPos();
	}

	@Override
	public int read() throws IOException {
		return inputStream.read();
	}

	@Override
	public int read(@Nonnull byte[] buffer, int offset, int length) throws IOException {
		return inputStream.read(buffer, offset, length);
	}

	/**
	 * instead by readByteBuffer.
	 */
	@Deprecated
	public int read(int position, @Nonnull byte[] buffer, int offset, int length) throws IOException {
		if (fileSystem.isDistributedFS()) {
			return ((HadoopDataInputStream) inputStream).getHadoopInputStream().read(position, buffer, offset, length);
		} else {
			ByteBuffer bb = ByteBuffer.wrap(buffer, offset, length);
			return ((GeminiLocalByteBufferableInputStream) inputStream).
				read(position, bb);
		}
	}

	public int readByteBuffer(int position, @Nonnull ByteBuffer buffer, int offset, int length) throws IOException {
		if (fileSystem.isDistributedFS()) {
			throw new IOException("now GeminiInputStream HDFS not support readByteBuffer");
		} else {
			return ((GeminiLocalByteBufferableInputStream) inputStream).
				read(position, buffer);
		}
	}

	@Override
	public int available() throws IOException {
		return inputStream.available();
	}

	@Override
	public void seek(long desired) throws IOException {
		inputStream.seek(desired);
	}

	@Override
	public void close() throws IOException {
		inputStream.close();
	}
}
