/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.fs;

import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.fs.hdfs.HadoopFileSystem;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * GeminiOutputStream.
 */
public class GeminiOutputStream extends FSDataOutputStream {

	private static final Logger LOG = LoggerFactory.getLogger(GeminiOutputStream.class);

	private static final int DEFAULT_BUFFER_SIZE = 4 * 1024;

	private final FileSystem fileSystem;

	private final Path filePath;

	private FSDataOutputStream outStream;

	/** lock for all operations to avoid concurrent access the underlying outputstream.*/
	private final Object lock = new Object();

	public GeminiOutputStream(
		Path filePath
	) throws IOException {
		this(filePath, FileSystem.WriteMode.NO_OVERWRITE);
	}

	public GeminiOutputStream(
		Path filePath,
		FileSystem.WriteMode writeMode
	) throws IOException {
		this(filePath, writeMode, DEFAULT_BUFFER_SIZE);
	}

	public GeminiOutputStream(
		Path filePath,
		FileSystem.WriteMode writeMode,
		int bufferSize
		) throws IOException {
		this.filePath = filePath;
		this.fileSystem = FileSystem.getUnguardedFileSystem(filePath.toUri());

		FSDataOutputStream stream = fileSystem.create(filePath, writeMode);
		// hdfs client has it's own buffer (default 4K), so we don't need buffer here
		if (fileSystem instanceof HadoopFileSystem) {
			this.outStream = stream;
		} else {
			this.outStream = new GeminiBufferedOutputStream(stream, bufferSize);
		}
	}

	@Override
	public void write(int b) throws IOException {
		synchronized (lock) {
			outStream.write(b);
		}
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		synchronized (lock) {
			outStream.write(b, off, len);
		}
	}

	@Override
	public long getPos() throws IOException {
		synchronized (lock) {
			return outStream.getPos();
		}
	}

	@Override
	public void flush() throws IOException {
		synchronized (lock) {
			outStream.flush();
		}
	}

	@Override
	public void sync() throws IOException {
		synchronized (lock) {
			outStream.sync();
		}
	}

	@Override
	public void close() throws IOException {
		synchronized (lock) {
			outStream.close();
		}
	}
}
