/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.runtime.state.gemini.engine.AbstractGeminiKMap;
import org.apache.flink.runtime.state.gemini.engine.GRegionContext;
import org.apache.flink.runtime.state.gemini.engine.GRegionID;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.memstore.WriteBufferKMap;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;
import org.apache.flink.runtime.state.gemini.engine.page.PageIndex;
import org.apache.flink.runtime.state.gemini.engine.page.PageSerdeFlink2Key;
import org.apache.flink.runtime.state.gemini.engine.page.PageStoreKMap;
import org.apache.flink.util.Preconditions;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import javax.annotation.Nullable;

import java.util.Map;

/**
 * AbstractGRegionKMapImpl.
 * K is type of key.
 * V is type of value.
 */
public abstract class AbstractGRegionKMapImpl<K, MK, MV, M extends Map<MK, MV>> implements AbstractGeminiKMap<K, MK, MV, M>, HashGRegion {

	protected final GContext gContext;

	protected final GRegionContext gRegionContext;

	protected WriteBufferKMap<K, MK, MV> writeBuffer;

	protected PageStoreKMap<K, MK, MV> pageStore;

	protected final EventExecutor regionEventExecutor;

	protected boolean readCopy;

	protected TypeSerializer<K> keySerializer;

	protected TypeSerializer<MK> mkSerializer;

	protected TypeSerializer<MV> mvSerializer;

	public AbstractGRegionKMapImpl(GRegionContext gRegionContext,
		@Nullable PageIndex pageIndex) {
		this.gRegionContext = Preconditions.checkNotNull(gRegionContext);
		this.gContext = gRegionContext.getGContext();

		// TODO how to choose an event executor
		this.regionEventExecutor = gContext.getSupervisor().getRegionExecutorGroup().next();
		gRegionContext.getPageStoreStats().setRegionExecutor(regionEventExecutor);
		gContext.getSupervisor().getCacheManager().addRegionEventExecutor(this.regionEventExecutor);
		this.readCopy = gRegionContext.getGContext().getGConfiguration().isReadCopy();
		this.keySerializer = gRegionContext.getPageSerdeFlink().getKeySerde();
		this.mkSerializer = ((PageSerdeFlink2Key) gRegionContext.getPageSerdeFlink()).getKey2Serde();
		this.mvSerializer = gRegionContext.getPageSerdeFlink().getValueSerde();
		init(pageIndex);
	}

	abstract void init(PageIndex pageIndex);

	@Override
	public void remove(K key) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		writeBuffer.removeKey(key);
	}

	@Override
	public void removeAll() {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		long lastSeqID = gRegionContext.getLastSeqID();
		gRegionContext.setRemoveAllSeqID(lastSeqID);
		writeBuffer.reset();
	}

	@Override
	public Iterable<K> keys() {
		gContext.checkDBStatus();
		return getAll().keySet();
	}

	@Override
	public boolean contains(K key) {
		gContext.checkDBStatus();
		//TODO: current writeBuffer.contains may return wrong answer
		// we get the whole state here to correct the logic,
		// but it doesn't have the best performance.
		gContext.incAccessNumber();
		M result = internalGet(key, false);
		if (result == null || result.isEmpty()) {
			return false;
		} else {
			return true;
		}
	}

	@Override
	public boolean contains(K key, MK mapKey) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		return internalGet(key, mapKey, false) != null;
	}

	@Override
	public MV get(K key, MK mapKey) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		return internalGet(key, mapKey, true);
	}

	@Override
	public MV getOrDefault(K key, MK mapKey, MV defaultMapValue) {
		gContext.checkDBStatus();
		MV result = get(key, mapKey);
		return result == null ? defaultMapValue : result;
	}

	@Override
	public void add(K key, MK mapKey, MV mapValue) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		writeBuffer.add(key, mapKey, mapValue);
	}

	@Override
	public void addAll(K key, Map<? extends MK, ? extends MV> mappings) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		writeBuffer.add(key, (Map<MK, MV>) mappings);
	}

	@Override
	public void remove(K key, MK mapKey) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		writeBuffer.remove(key, mapKey);
	}

	@Override
	public GRegionContext getGRegionContext() {
		return this.gRegionContext;
	}

	@Override
	public PageStoreKMap<K, MK, MV> getPageStore() {
		return pageStore;
	}

	@Override
	public GRegionID getRegionId() {
		return gRegionContext.getRegionId();
	}

	@Override
	public WriteBufferKMap<K, MK, MV> getWriteBuffer() {
		return writeBuffer;
	}

	@Override
	public EventExecutor getExecutor() {
		return this.regionEventExecutor;
	}

	private MV internalGet(K key, MK mapKey, boolean checkReadCopy) {
		GSValue<MV> value = writeBuffer.get(key, mapKey);
		if (value != null) {
			if (value.getValueType() == GValueType.Delete || gRegionContext.filterState(value.getSeqID())) {
				return null;
			}
			return checkReadCopy ? copyMVIfNeeded(value.getValue()) : value.getValue();
		}
		return pageStore.get(key, mapKey);
	}

	protected abstract M internalGet(K key, boolean checkReadCopy);

	protected K copyKeyIfNeeded(K key) {
		return readCopy ? keySerializer.copy(key) : key;
	}

	protected MK copyMKIfNeeded(MK mk) {
		return readCopy ? mkSerializer.copy(mk) : mk;
	}

	protected MV copyMVIfNeeded(MV mv) {
		return readCopy ? mvSerializer.copy(mv) : mv;
	}
}
