/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.api.common.typeutils.base.ByteSerializer;
import org.apache.flink.runtime.state.KeyGroupRangeAssignment;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.GRegionIDImpl;
import org.apache.flink.runtime.state.gemini.engine.GTableDescription;
import org.apache.flink.runtime.state.gemini.engine.GeminiPKey2;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;
import org.apache.flink.runtime.state.gemini.engine.page.PKey2Serializer;
import org.apache.flink.runtime.state.gemini.engine.page.PageSerdeFlink2KeyImpl;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import static org.apache.flink.runtime.state.gemini.engine.GRegionID.G_REGION_INDEX_1;

/**
 * AbstractGTableSubKeyedMapImpl.
 * @param <K>
 * @param <N>
 * @param <MK>
 * @param <MV>
 */
public abstract class AbstractGTableSubKeyedMapImpl<K, N, MK, MV> implements GTableWithPKey2<K, N> {
	protected AbstractGRegionKMapImpl<GeminiPKey2<K, N>, MK, MV, ? extends Map<MK, MV>>[] regions;

	private final KMapTableDescription<K, N, Byte> indexDescription;
	GRegionKMapImpl<K, N, Byte>[] keyIndexRegions;

	int startGroup;
	int numberGroups;
	int endGroup;
	int maxParallelism;
	GTableDescription description;
	GContext context;

	public AbstractGTableSubKeyedMapImpl(
		GTableDescription description,
		int startGroup,
		int numberGroups,
		int maxParallelism,
		GContext context) {
		this.startGroup = startGroup;
		this.numberGroups = numberGroups;
		this.endGroup = startGroup + numberGroups;
		this.maxParallelism = maxParallelism;
		this.description = description;
		this.context = context;

		this.regions = new AbstractGRegionKMapImpl[numberGroups];
		this.keyIndexRegions = new GRegionKMapImpl[numberGroups];

		PageSerdeFlink2KeyImpl<GeminiPKey2<K, N>, MK, MV> pageSerde = (PageSerdeFlink2KeyImpl<GeminiPKey2<K, N>, MK, MV>) description.getPageSerde();
		PKey2Serializer<K, N> pKey2Serializer = (PKey2Serializer<K, N>) pageSerde.getKeySerde();
		PageSerdeFlink2KeyImpl<K, N, Byte> indexPageSerde = new PageSerdeFlink2KeyImpl<>(
			pKey2Serializer.getFirstSerializer(),
			pKey2Serializer.getSecondSerializer(),
			ByteSerializer.INSTANCE,
			null,
			context.getGConfiguration().isChecksumEnable());
		this.indexDescription = new KMapTableDescription<>(
			description.getTableName(),
			startGroup,
			numberGroups,
			maxParallelism,
			indexPageSerde);
	}

	@Override
	public GTableDescription getTableDescription() {
		return description;
	}

	@Override
	public Iterator<N> getSecondaryKeyByFirstKey(K key) {
		int group = KeyGroupRangeAssignment.assignToKeyGroup(key, maxParallelism);
		int offset = group - startGroup;
		GRegionKMapImpl<K, N, Byte> indexRegion = keyIndexRegions[offset];
		if (indexRegion == null) {
			return Collections.emptyIterator();
		}

		AbstractGRegionKMapImpl<GeminiPKey2<K, N>, MK, MV, ? extends Map<MK, MV>> dataRegion = regions[offset];
		Map<N, Byte> indexMap = indexRegion.get(key);
		return indexMap == null ? Collections.emptyIterator() : indexMap.keySet()
			.stream()
			.filter(x -> dataRegion.contains(getGeminiPKey2(key, x)))
			.iterator();
	}

	@Override
	public GeminiPKey2<K, N> getGeminiPKey2(K key1, N key2) {
		return new GeminiPKey2<>(key1, key2);
	}

	public GRegionKMapImpl<K, N, Byte> getIndexRegion(K key) {
		int group = KeyGroupRangeAssignment.assignToKeyGroup(key, maxParallelism);
		int offset = group - startGroup;
		GRegionKMapImpl region = keyIndexRegions[offset];

		if (region == null) {
			//now we only support 1 index
			region = (GRegionKMapImpl<K, N, Byte>) indexDescription.createRegion(context,
				this,
				new GRegionIDImpl(G_REGION_INDEX_1, group));
			keyIndexRegions[offset] = region;
		}
		return region;
	}

	@Override
	public Iterator<GRegion> regionIterator() {
		return new Iterator<GRegion>() {
			private int currentIndex = 0;

			private void advance() {
				currentIndex++;
				while (currentIndex < regions.length && regions[currentIndex] == null) {
					currentIndex++;
				}
			}

			@Override
			public boolean hasNext() {
				if (currentIndex < regions.length && regions[currentIndex] != null) {
					return true;
				}

				advance();
				return currentIndex < regions.length;
			}

			@Override
			public GRegion next() {
				return regions[currentIndex++];
			}
		};
	}

	@Override
	public Iterator<GRegion> indexRegionIterator() {
		return new Iterator<GRegion>() {
			private int currentIndex = 0;

			private void advance() {
				currentIndex++;
				while (currentIndex < keyIndexRegions.length && keyIndexRegions[currentIndex] == null) {
					currentIndex++;
				}
			}

			@Override
			public boolean hasNext() {
				if (currentIndex < keyIndexRegions.length && keyIndexRegions[currentIndex] != null) {
					return true;
				}

				advance();
				return currentIndex < keyIndexRegions.length;
			}

			@Override
			public GRegion next() {
				return keyIndexRegions[currentIndex++];
			}
		};
	}

	@Override
	public GTableDescription getIndexDescription() {
		return indexDescription;
	}
}

