/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.runtime.state.gemini.engine.GRegionContext;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValueMap;
import org.apache.flink.runtime.state.gemini.engine.memstore.WriteBufferKMapHashImpl;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;
import org.apache.flink.runtime.state.gemini.engine.page.PageIndex;
import org.apache.flink.runtime.state.gemini.engine.page.PageStoreHashKMapImpl;

import org.apache.flink.shaded.guava18.com.google.common.collect.Maps;

import javax.annotation.Nullable;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.stream.Collectors;

/**
 * GRegionKMapImpl.
 * K is type of key.
 * V is type of value.
 */
public class GRegionKMapImpl<K, MK, MV> extends AbstractGRegionKMapImpl<K, MK, MV, Map<MK, MV>> {

	public GRegionKMapImpl(GRegionContext gRegionContext) {
		super(gRegionContext, null);
	}

	public GRegionKMapImpl(GRegionContext gRegionContext, PageIndex pageIndex) {
		super(gRegionContext, pageIndex);
	}

	@Override
	void init(PageIndex pageIndex) {
		this.pageStore = new PageStoreHashKMapImpl<>(this, pageIndex, regionEventExecutor);
		this.writeBuffer = new WriteBufferKMapHashImpl<>(this, regionEventExecutor, pageStore);
	}

	@Override
	public void put(K key, Map<MK, MV> value) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		long seqID = gRegionContext.getNextSeqID();
		writeBuffer.put(key, Maps.transformEntries(value,
			(mk, mv) -> GSValue.of(mv, GValueType.PutValue, seqID)));
	}

	@Override
	public Map<MK, MV> get(K key) {
		gContext.checkDBStatus();
		gContext.incAccessNumber();
		return internalGet(key, true);
	}

	@Nullable
	private Map<MK, MV> mergeTwoMap(@Nullable Map<MK, GSValue<MV>> newer, Map<MK, GSValue<MV>> older, boolean checkReadCopy) {
		Map<MK, MV> ret = new HashMap<>();
		if (older != null) {
			ret = older.entrySet().stream()
				.filter(e -> !GValueType.Delete.equals(e.getValue().getValueType()) && !gRegionContext.filterState(e.getValue().getSeqID()))
				.collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().getValue()));
		}
		if (newer != null) {
			for (Map.Entry<MK, GSValue<MV>> entry : newer.entrySet()) {
				if (gRegionContext.filterState(entry.getValue().getSeqID())) {
					continue;
				}
				if (entry.getValue().getValueType().equals(GValueType.Delete)) {
					ret.remove(entry.getKey());
				} else {
					MK mk = entry.getKey();
					MV mv = entry.getValue().getValue();
					ret.put(checkReadCopy ? copyMKIfNeeded(mk) : mk, checkReadCopy ? copyMVIfNeeded(mv) : mv);
				}
			}
		}

		return ret.isEmpty() ? null : ret;
	}

	private Map<MK, MV> genMapFromGSValueMap(GSValueMap<MK, MV> gsValueMap, boolean checkReadCopy) {
		Map<MK, GSValue<MV>> valueMap = gsValueMap.getValue();
		Map<MK, MV> ret = new HashMap<>();
		if (valueMap != null) {
			for (SortedMap.Entry<MK, GSValue<MV>> entry : valueMap.entrySet()) {
				if (!GValueType.Delete.equals(entry.getValue().getValueType()) &&
					!gRegionContext.filterState(entry.getValue().getSeqID())) {
					MK mk = entry.getKey();
					MV mv = entry.getValue().getValue();
					ret.put(checkReadCopy ? copyMKIfNeeded(mk) : mk, checkReadCopy ? copyMVIfNeeded(mv) : mv);
				}
			}
		}
		return ret.isEmpty() ? null : ret;
	}

	@Override
	public Map<MK, MV> getOrDefault(K key, Map<MK, MV> defaultValue) {
		gContext.checkDBStatus();
		Map<MK, MV> result = get(key);
		return (result == null || result.isEmpty()) ? defaultValue : result;
	}

	@Override
	public Map<K, Map<MK, MV>> getAll() {
		gContext.checkDBStatus();
		Map<K, Map<MK, MV>> results = new HashMap<>();
		getAll(results);
		return results;
	}

	@Override
	public void getAll(Map<K, Map<MK, MV>> results) {
		gContext.checkDBStatus();
		//TODO provide better performance.
		gContext.incAccessNumber();
		Set<K> allKeysIncludeDelete = new HashSet<>();
		writeBuffer.allKeysIncludeDeleted(allKeysIncludeDelete);
		pageStore.allKeysIncludeDeleted(allKeysIncludeDelete);

		for (K key : allKeysIncludeDelete) {
			Map<MK, MV> mapResult = get(key);

			if (mapResult != null) {
				results.put(copyKeyIfNeeded(key), mapResult);
			}
		}
	}

	@Override
	protected Map<MK, MV> internalGet(K key, boolean checkReadCopy) {
		GSValueMap<MK, MV> gsValueMap = writeBuffer.get(key);
		if (gsValueMap != null) {
			switch (gsValueMap.getValueType()) {
				case Delete:
					return null;
				case PutMap:
					return genMapFromGSValueMap(gsValueMap, checkReadCopy);
				default:
					// nothing need to do.
			}
		}

		Map<MK, GSValue<MV>> mapResult = pageStore.get(key);
		return mergeTwoMap(gsValueMap == null ? null : gsValueMap.getValue(), mapResult, checkReadCopy);
	}
}
