/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.runtime.state.KeyGroupRangeAssignment;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.GRegionIDImpl;
import org.apache.flink.runtime.state.gemini.engine.GTableDescription;
import org.apache.flink.runtime.state.gemini.engine.GeminiPKey2;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;

import java.util.Map;

import static org.apache.flink.runtime.state.gemini.engine.GRegionID.G_REGION_DATA;
import static org.apache.flink.util.Preconditions.checkArgument;

/**
 * GeminiTableSubKeyValueImpl.
 * K is key.
 * N is namespace.
 * MK,MV depicts user Map.
 */
public class GTableSubKeyedMapImpl<K, N, MK, MV> extends AbstractGTableSubKeyedMapImpl<K, N, MK, MV> {

	public GTableSubKeyedMapImpl(
		GTableDescription description,
		int startGroup,
		int numberGroups,
		int maxParallelism,
		GContext context) {
		super(description, startGroup, numberGroups, maxParallelism, context);
	}

	@SuppressWarnings("unchecked")
	@Override
	public AbstractGRegionKMapImpl<GeminiPKey2<K, N>, MK, MV, ? extends Map<MK, MV>> getRegion(GeminiPKey2<K, N> key) {
		// we use the first key (K) as the input of assignToKeyGroup, so the group is same as the DataStream#KeyBy()
		int group = KeyGroupRangeAssignment.assignToKeyGroup(key.getFirstKey(), maxParallelism);
		int offset = group - startGroup;
		AbstractGRegionKMapImpl<GeminiPKey2<K, N>, MK, MV, ? extends Map<MK, MV>> region = regions[offset];

		if (region == null) {
			//indexID 0 means data region.
			region = (GRegionKMapImpl<GeminiPKey2<K, N>, MK, MV>) description.createRegion(context,
				this,
				new GRegionIDImpl(G_REGION_DATA, group));
			regions[offset] = region;
		}

		return region;
	}

	@Override
	public void setRegion(int idx, GRegion region) {
		checkArgument(idx >= startGroup && idx < endGroup);
		regions[idx - startGroup] = (GRegionKMapImpl<GeminiPKey2<K, N>, MK, MV>) region;
	}

	@Override
	public void setIndexRegion(int idx, GRegion indexRegion) {
		checkArgument(idx >= startGroup && idx < endGroup);
		keyIndexRegions[idx - startGroup] = (GRegionKMapImpl<K, N, Byte>) indexRegion;
	}
}
