/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.runtime.state.KeyGroupRangeAssignment;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.GRegionID;
import org.apache.flink.runtime.state.gemini.engine.GRegionIDImpl;
import org.apache.flink.runtime.state.gemini.engine.GTable;
import org.apache.flink.runtime.state.gemini.engine.GTableDescription;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;

import java.util.Collections;
import java.util.Iterator;

import static org.apache.flink.util.Preconditions.checkArgument;
import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * GeminiTableSubKeyValueImpl.
 * K is key
 * N is namespace
 * V can only be Value
 */
public class GTableSubKeyedValueImpl<K, N, V> implements GTable<K> {
	private GRegionKMapImpl<K, N, V>[] regions;

	private final int startGroup;

	private final int numberGroups;

	private final int endGroup;
	private final int maxParallelism;

	private final GTableDescription description;

	private final GContext gContext;

	public GTableSubKeyedValueImpl(
		GTableDescription description,
		int startGroup,
		int numberGroups,
		int maxParallelism,
		GContext context) {
		checkArgument(startGroup >= 0);
		checkArgument(numberGroups > 0);
		checkArgument(maxParallelism > 0);
		checkArgument(maxParallelism >= numberGroups, "Max parallelism must bigger thant groups of current table.");

		this.startGroup = startGroup;
		this.numberGroups = numberGroups;
		this.endGroup = startGroup + numberGroups;
		this.maxParallelism = maxParallelism;
		this.description = checkNotNull(description);
		this.gContext = checkNotNull(context);
		this.regions = new GRegionKMapImpl[numberGroups];
	}

	@Override
	public GTableDescription getTableDescription() {
		return description;
	}

	@SuppressWarnings("unchecked")
	@Override
	public GRegionKMapImpl<K, N, V> getRegion(K key) {
		int group = KeyGroupRangeAssignment.assignToKeyGroup(key, maxParallelism);
		int offset = group - startGroup;
		GRegionKMapImpl<K, N, V> region = regions[offset];

		if (region == null) {
			region = (GRegionKMapImpl<K, N, V>) description.createRegion(gContext,
				this,
				new GRegionIDImpl(GRegionID.G_REGION_DATA, group));
			regions[offset] = region;
		}

		return region;
	}

	@Override
	public Iterator<GRegion> regionIterator() {
		return new Iterator<GRegion>() {

			private int currentIndex = 0;

			private void advance() {
				currentIndex++;
				while (currentIndex < regions.length && regions[currentIndex] == null) {
					currentIndex++;
				}
			}

			@Override
			public boolean hasNext() {
				if (currentIndex < regions.length && regions[currentIndex] != null) {
					return true;
				}

				advance();
				return currentIndex < regions.length;
			}

			@Override
			public GRegion next() {
				return regions[currentIndex++];
			}
		};
	}

	@Override
	public Iterator<GRegion> indexRegionIterator() {
		return Collections.emptyIterator();
	}

	@Override
	public void setRegion(int idx, GRegion region) {
		checkArgument(idx >= startGroup && idx < endGroup);
		regions[idx - startGroup] = (GRegionKMapImpl<K, N, V>) region;
	}

	@Override
	public void setIndexRegion(int idx, GRegion indexRegion) {
		throw new UnsupportedOperationException("");
	}

	@Override
	public GTableDescription getIndexDescription() {
		throw new UnsupportedOperationException();
	}
}
