/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.hashtable;

import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.GRegionContext;
import org.apache.flink.runtime.state.gemini.engine.GRegionID;
import org.apache.flink.runtime.state.gemini.engine.GTable;
import org.apache.flink.runtime.state.gemini.engine.GTableDescription;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;
import org.apache.flink.runtime.state.gemini.engine.page.PageIndex;
import org.apache.flink.runtime.state.gemini.engine.page.PageSerdeFlink;
import org.apache.flink.util.Preconditions;

/**
 * KVTableDescription.
 * @param <K>
 * @param <S>
 */
public class KVTableDescription<K, S> implements GTableDescription {

	private final String tableName;

	private final int startGroup;

	private final int numberGroups;

	private final int maxParallelism;

	private final PageSerdeFlink<K, S> pageSerdeFlink;

	public KVTableDescription(
		String tableName,
		int startGroup,
		int numberGroups,
		int maxParallelism,
		PageSerdeFlink<K, S> pageSerdeFlink
	) {
		this.tableName = Preconditions.checkNotNull(tableName);
		this.startGroup = startGroup;
		this.numberGroups = numberGroups;
		this.maxParallelism = maxParallelism;
		this.pageSerdeFlink = pageSerdeFlink;
	}

	@Override
	public String getTableName() {
		return tableName;
	}

	@Override
	public GTable createTable(GContext gContext) {
		return new GTableOneKeyImpl<K, S>(
			this,
			startGroup,
			numberGroups,
			maxParallelism,
			gContext
		);
	}

	@Override
	public GRegion createRegion(GContext gContext, GTable gTable, GRegionID regionId) {
		GRegionContext regionContext = new GRegionContext(gContext, tableName, regionId, pageSerdeFlink);
		return new GRegionKVImpl<K, S>(regionContext);
	}

	@Override
	public GRegion createRegion(
		GContext gContext, GTable gTable, GRegionID regionId, PageIndex pageIndex) {
		GRegionContext regionContext = new GRegionContext(gContext, tableName, regionId, pageSerdeFlink);
		return new GRegionKVImpl<K, S>(regionContext, pageIndex);
	}

	@Override
	public GRegion createRegion(
		GRegionContext regionContext, PageIndex pageIndex) {
		return new GRegionKVImpl<K, S>(regionContext, pageIndex);
	}

	@Override
	public PageSerdeFlink getPageSerde() {
		return null;
	}
}
