/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.handler.PageHandler;
import org.apache.flink.runtime.state.gemini.engine.handler.PageKMapHandlerImpl;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;
import org.apache.flink.runtime.state.gemini.engine.page.PageStore;
import org.apache.flink.runtime.state.gemini.engine.page.PageStoreKMap;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedDeque;

/**
 * AbstractWriteBufferKMapHashImpl.
 */
public abstract class AbstractWriteBufferKMapHashImpl<K, MK, MV> extends AbstractWriteBuffer<K, Map<MK, GSValue<MV>>> implements WriteBufferKMap<K, MK, MV> {
	private static final Logger LOG = LoggerFactory.getLogger(AbstractWriteBufferKMapHashImpl.class);
	protected SegmentKMap<K, MK, MV> active;
	protected ConcurrentLinkedDeque<SegmentKMap<K, MK, MV>> snapshotQueue = new ConcurrentLinkedDeque<>();

	public AbstractWriteBufferKMapHashImpl(
		GRegion gRegion, EventExecutor eventExecutor, PageStore pageStore) {
		super(gRegion, eventExecutor, pageStore);
	}

	abstract void initActive();

	abstract Map<MK, GSValue<MV>> createPOJOMap();

	@Override
	PageHandler createPageHandler(Segment segment, boolean onlyEstimatedSize) {
		return new PageKMapHandlerImpl<>(gRegionContext,
			(PageStoreKMap<K, MK, MV>) gRegion.getPageStore(),
			(SegmentKMapImpl<K, MK, MV>) segment,
			onlyEstimatedSize);
	}

	@Override
	public Segment getActiveSegment() {
		return active;
	}

	@Override
	Segment pollFlushingSegment() {
		return snapshotQueue.poll();
	}

	@Override
	public void add(K key, MK mkey, MV mvalue) {
		active.add(key, mkey, mvalue);
		checkResource();
	}

	@Override
	public void add(K key, Map<MK, MV> map) {
		active.add(key, map);
		checkResource();
	}

	@Override
	public void remove(K key, MK mapKey) {
		active.remove(key, mapKey);
		checkResource();
	}

	@Override
	public GValueType contains(K key, MK mapKey) {
		GSValueMap<MK, MV> mapValue = active.get(key);
		if (mapValue != null) {
			mapValue.requestCount++;
			if (mapValue.valueType == GValueType.Delete) {
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return GValueType.Delete;
			}
			GSValue<MV> value = mapValue.getValue().get(mapKey);
			if (value != null) {
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return value.valueType;
			}
		}
		Iterator<SegmentKMap<K, MK, MV>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKMap<K, MK, MV> inactive = iterator.next();
			mapValue = inactive.get(key);
			if (mapValue != null) {
				mapValue.requestCount++;
				if (mapValue.valueType == GValueType.Delete) {
					gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
					return GValueType.Delete;
				}
				GSValue<MV> value = mapValue.getValue().get(mapKey);
				if (value != null) {
					gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
					return value.valueType;
				}
			}
		}
		gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferMissCount();
		return null;
	}

	@Override
	public void put(K key, Map<MK, GSValue<MV>> value) {
		active.put(key, value);
		checkResource();
	}

	@Override
	public GSValue<MV> get(K key, MK mapKey) {
		GSValue<MV> gsValue = active.get(key, mapKey);
		if (gsValue != null) {
			gsValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			return gsValue;
		}
		Iterator<SegmentKMap<K, MK, MV>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKMap<K, MK, MV> inactive = iterator.next();
			gsValue = inactive.get(key, mapKey);
			if (gsValue != null) {
				gsValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return gsValue;
			}
		}
		gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferMissCount();
		return null;
	}

	@Override
	public GSValueMap<MK, MV> get(K key) {
		List<GSValueMap<MK, MV>> reverseOrderList = new ArrayList<>();
		GSValueMap<MK, MV> mapValue = active.get(key);
		if (mapValue != null) {
			mapValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			if (mapValue.valueType == GValueType.PutMap) {
				return mapValue;
			} else if (mapValue.valueType == GValueType.Delete) {
				return mapValue;
			}
			reverseOrderList.add(mapValue);
		}
		Iterator<SegmentKMap<K, MK, MV>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKMap<K, MK, MV> inactive = iterator.next();
			mapValue = inactive.get(key);
			if (mapValue != null) {
				mapValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				reverseOrderList.add(mapValue);
				if (mapValue.valueType == GValueType.PutMap || mapValue.valueType == GValueType.Delete) {
					break;
				}
			}
		}
		return reverseOrderList.size() == 0 ? null : mergeGSValueMap(reverseOrderList);
	}

	@Override
	public void getAll(Map<K, GSValue<Map<MK, GSValue<MV>>>> container) {
		// TODO
	}

	@Override
	public void allKeysIncludeDeleted(Set<K> container) {
		Segment activeMap = getActiveSegment();
		container.addAll(activeMap.getData().keySet());

		Iterator<SegmentKMap<K, MK, MV>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKMap<K, MK, MV> inactive = iterator.next();
			container.addAll(inactive.getData().keySet());
		}
	}

	@Override
	public void removeKey(K key) {
		active.removeKey(key);
		checkResource();
	}

	@Override
	public void reset() {
		initActive();
	}

	@Override
	public GValueType contains(K key) {
		// TODO: may be return wrong value, by pass to region.
		GSValueMap<MK, MV> mapValue = active.get(key);
		if (mapValue != null) {
			mapValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			return mapValue.valueType;
		}
		Iterator<SegmentKMap<K, MK, MV>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKMap<K, MK, MV> inactive = iterator.next();
			mapValue = inactive.get(key);
			if (mapValue != null) {
				mapValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return mapValue.valueType;
			}
		}
		gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferMissCount();
		return null;
	}

	private GSValueMap<MK, MV> mergeGSValueMap(List<GSValueMap<MK, MV>> reverseOrderList) {
		if (reverseOrderList == null || reverseOrderList.size() == 0) {
			return null;
		}
		//TODO check TTL
		int index = reverseOrderList.size() - 1;
		//Value list is right order.
		Map<MK, GSValue<MV>> newMap = createPOJOMap();
		//judge a compacted map value's type is a little complicated.
		// any -> Delete -> AddMap -> AddMap, final type must be PutMap
		// any -> PutMap -> AddMap -> AddMap, final type must be PutMap
		// AddMap -> AddMap -> AddMap, final type must be AddMap
		GValueType finalGValueType = GValueType.AddMap;
		//map value's seqID is newest. that will be more confused.
		long seqID = reverseOrderList.get(0).getSeqID();
		while (index >= 0) {
			GSValueMap<MK, MV> currentMap = reverseOrderList.get(index);
			if (currentMap.getValueType() == GValueType.Delete) {
				newMap.clear();
				finalGValueType = GValueType.PutMap;
			} else if (currentMap.getValueType() == GValueType.PutMap) {
				newMap.clear();
				newMap.putAll(currentMap.value);
				finalGValueType = GValueType.PutMap;
			} else {
				newMap.putAll(currentMap.value);
			}
			index--;
		}

		return new GSValueMap<>(newMap, finalGValueType, seqID);
	}
}
