/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.runtime.state.gemini.engine.page.GValueType;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.Objects;

/**
 * GSValue.
 */
public class GSValue<V> {
	protected final GValueType valueType;
	protected V value;
	protected int requestCount;
	protected long seqID;

	public GSValue(V value, GValueType gValueType, long seqID) {
		this.valueType = gValueType;
		this.value = value;
		this.seqID = seqID;
	}

	public GValueType getValueType() {
		return valueType;
	}

	public V getValue() {
		return value;
	}

	public V setValue(V newValue) {
		V oldValue = value;
		value = newValue;
		return oldValue;
	}

	public int getRequestCount() {
		return requestCount;
	}

	public long getSeqID() {
		return seqID;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		GSValue<?> gsValue = (GSValue<?>) o;
		boolean result = valueType == gsValue.valueType;
		if (!result) {
			return false;
		}
		result = seqID == gsValue.seqID;

		if (!result) {
			return false;
		}

		return valueType == GValueType.Delete || (value == gsValue.value || value.equals(gsValue.value));
	}

	@Override
	public int hashCode() {
		return Objects.hash(valueType, value, seqID);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("valueType", valueType).
			add("value", value).
			add("requestCount", requestCount).
			add("seqID", seqID).toString();
	}

	public static <V> GSValue<V> of(V value, GValueType valueType, long seqID) {
		return new GSValue<>(value, valueType, seqID);
	}

	public static <V> GSValue<V> of(
		V value,
		GValueType valueType,
		long seqID,
		int requestCount) {
		GSValue gsValue = new GSValue<>(value, valueType, seqID);
		gsValue.requestCount = requestCount;
		return gsValue;
	}
}
