/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.runtime.state.gemini.engine.GRegionContext;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;

import java.util.HashMap;
import java.util.Map;

/**
 * SegmentImpl.
 */
public class SegmentImpl<K, V> implements Segment<K, V> {

	private final long segmentID;
	private final GRegionContext gRegionContext;
	private final Map<K, GSValue<V>> dataMap;
	private final long version;
	private int recordCount = 0;

	private boolean writeCopy;

	private TypeSerializer<K> keySerializer;

	private TypeSerializer<V> valueSerializer;

	@SuppressWarnings("unchecked")
	public SegmentImpl(long segmentID, GRegionContext gRegionContext) {
		this.segmentID = segmentID;
		this.gRegionContext = gRegionContext;
		this.dataMap = new HashMap<>();
		this.version = gRegionContext.getGContext().getCurVersion();
		this.writeCopy = gRegionContext.getGContext().getGConfiguration().isWriteCopy();
		this.keySerializer = gRegionContext.getPageSerdeFlink().getKeySerde();
		this.valueSerializer = gRegionContext.getPageSerdeFlink().getValueSerde();
	}

	@SuppressWarnings("unchecked")
	public SegmentImpl(long segmentID, GRegionContext gRegionContext, Map<K, GSValue<V>> dataMap) {
		this.segmentID = segmentID;
		this.gRegionContext = gRegionContext;
		this.dataMap = dataMap;
		this.version = gRegionContext.getGContext().getCurVersion();
		this.writeCopy = gRegionContext.getGContext().getGConfiguration().isWriteCopy();
		this.keySerializer = gRegionContext.getPageSerdeFlink().getKeySerde();
		this.valueSerializer = gRegionContext.getPageSerdeFlink().getValueSerde();
	}

	@Override
	public void put(K key, V value) {
		// TODO no need to copy key if the mapping has existed, but how to do this in an efficient way
		if (dataMap.put(copyKeyIfNeeded(key), GSValue.of(copyValueIfNeeded(value), GValueType.PutValue, gRegionContext.getNextSeqID())) == null) {
			recordCount++;
			gRegionContext.getWriteBufferStats().addTotalRecordCount(1);
		}
	}

	@Override
	public GSValue<V> get(K key) {
		return dataMap.get(key);
	}

	@Override
	public void removeKey(K key) {
		// TODO no need to copy key if the mapping has existed, but how to do this in an efficient way
		dataMap.put(copyKeyIfNeeded(key), GSValue.of(null, GValueType.Delete, gRegionContext.getNextSeqID()));
	}

	@Override
	public long getSegmentID() {
		return this.segmentID;
	}

	@Override
	public int getRecordCount() {
		return recordCount;
	}

	@Override
	public long getVersion() {
		return this.version;
	}

	public Map<K, GSValue<V>> getDataMap() {
		return dataMap;
	}

	@Override
	public SegmentImpl<K, V> copySegment() {
		return new SegmentImpl<>(-1L, this.gRegionContext, new HashMap<>(dataMap));
	}

	@Override
	public Map getData() {
		return this.dataMap;
	}

	private K copyKeyIfNeeded(K key) {
		return writeCopy ? keySerializer.copy(key) : key;
	}

	private V copyValueIfNeeded(V value) {
		return writeCopy ? valueSerializer.copy(value) : value;
	}
}
