/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.handler.PageHandler;
import org.apache.flink.runtime.state.gemini.engine.handler.PageKValueHandlerImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKVImpl;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;
import org.apache.flink.runtime.state.gemini.engine.page.PageStore;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedDeque;

/**
 * WriteBufferHashImpl.
 */
public class WriteBufferHashImpl<K, V> extends AbstractWriteBuffer<K, V> {
	private static final Logger LOG = LoggerFactory.getLogger(WriteBufferHashImpl.class);
	private SegmentImpl<K, V> active;
	private ConcurrentLinkedDeque<Segment<K, V>> snapshotQueue = new ConcurrentLinkedDeque<>();

	public WriteBufferHashImpl(
		GRegion gRegion, EventExecutor eventExecutor, PageStore<K, V> pageStore) {
		super(gRegion, eventExecutor, pageStore);
		active = new SegmentImpl<>(segmentID++, gRegionContext);
	}

	@Override
	public Segment getActiveSegment() {
		return active;
	}

	@Override
	Segment addFlushingSegment() {
		Segment result = active;
		snapshotQueue.add(active);
		gRegionContext.getWriteBufferStats().addFlushingSegmentCount(1);
		active = new SegmentImpl<>(segmentID++, gRegionContext);
		return result;
	}

	@Override
	Segment pollFlushingSegment() {
		return snapshotQueue.poll();
	}

	@Override
	PageHandler createPageHandler(Segment segment, boolean onlyEstimatedSize) {
		return new PageKValueHandlerImpl<>((GRegionKVImpl<K, V>) gRegion,
			(SegmentImpl<K, V>) segment,
			onlyEstimatedSize);
	}

	@Override
	public void put(K key, V value) {
		active.put(key, value);
		checkResource();
	}

	@Override
	public GSValue<V> get(K key) {
		GSValue<V> gsValue = active.get(key);
		if (gsValue != null) {
			gsValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			return gsValue;
		}
		Iterator<Segment<K, V>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			Segment<K, V> inactive = iterator.next();
			gsValue = inactive.get(key);
			if (gsValue != null) {
				gsValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return gsValue;
			}
		}
		gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferMissCount();
		return null;
	}

	@Override
	public void getAll(Map<K, GSValue<V>> container) {
		Map<K, GSValue<V>> activeMap = active.getDataMap();
		for (Map.Entry<K, GSValue<V>> entry : activeMap.entrySet()) {
				container.putIfAbsent(entry.getKey(), entry.getValue());
		}

		Iterator<Segment<K, V>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			Segment<K, V> inactive = iterator.next();
			Map<K, GSValue<V>> inactiveMap = ((SegmentImpl) inactive).getDataMap();
			for (Map.Entry<K, GSValue<V>> entry : inactiveMap.entrySet()) {
				container.putIfAbsent(entry.getKey(), entry.getValue());
			}
		}
	}

	@Override
	public void allKeysIncludeDeleted(Set<K> container) {
		Segment activeMap = getActiveSegment();
		container.addAll(activeMap.getData().keySet());

		Iterator<Segment<K, V>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			Segment<K, V> inactive = iterator.next();
			container.addAll(inactive.getData().keySet());
		}
	}

	@Override
	public void removeKey(K key) {
		active.removeKey(key);
		checkResource();
	}

	@Override
	public void reset() {
		active = new SegmentImpl<>(segmentID++, gRegionContext);
	}

	@Override
	public GValueType contains(K key) {
		// TODO: may be return wrong answer, by pass to region.
		GSValue<V> getResult = get(key);
		if (getResult != null) {
			return getResult.valueType;
		}
		return null;
	}
}
