/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.handler.PageHandler;
import org.apache.flink.runtime.state.gemini.engine.handler.PageKListHandlerImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKListImpl;
import org.apache.flink.runtime.state.gemini.engine.page.GValueType;
import org.apache.flink.runtime.state.gemini.engine.page.PageStore;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedDeque;

/**
 * WriteBufferKListHashImpl.
 */
public class WriteBufferKListHashImpl<K, E> extends AbstractWriteBuffer<K, List<GSValue<E>>> implements WriteBufferKList<K, E> {
	private static final Logger LOG = LoggerFactory.getLogger(WriteBufferKListHashImpl.class);
	protected SegmentKList<K, E> active;
	protected ConcurrentLinkedDeque<SegmentKList<K, E>> snapshotQueue = new ConcurrentLinkedDeque<>();

	public WriteBufferKListHashImpl(
		GRegion gRegion, EventExecutor eventExecutor, PageStore pageStore) {
		super(gRegion, eventExecutor, pageStore);
		active = new SegmentKListImpl<>(segmentID++, gRegionContext);
	}

	@Override
	public void add(K key, E element) {
		active.add(key, element);
		checkResource();
	}

	@Override
	public void addAll(K key, Collection<? extends E> elements) {
		active.addAll(key, elements);
		checkResource();
	}

	@Override
	public void remove(K key, E element) {
		throw new GeminiRuntimeException("not support remove element");
	}

	@Override
	public void removeAll(K key, Collection<? extends E> elements) {
		active.removeAll(key, elements);
		checkResource();
	}

	@Override
	public Segment getActiveSegment() {
		return active;
	}

	@Override
	Segment pollFlushingSegment() {
		return snapshotQueue.poll();
	}

	@Override
	Segment addFlushingSegment() {
		Segment result = active;
		snapshotQueue.add(active);
		gRegionContext.getWriteBufferStats().addFlushingSegmentCount(1);
		active = new SegmentKListImpl<>(segmentID++, gRegionContext);
		return result;
	}

	@Override
	PageHandler createPageHandler(
		Segment segment, boolean onlyEstimatedSize) {
		return new PageKListHandlerImpl<>((GRegionKListImpl<K, E>) gRegion,
			(SegmentKListImpl<K, E>) segment,
			onlyEstimatedSize);
	}

	@Override
	public void put(K key, List<GSValue<E>> value) {
		active.put(key, value);
		checkResource();
	}

	@Override
	public GSValueList<E> get(K key) {
		List<GSValueList<E>> reverseOrderList = new ArrayList<>();
		GSValueList<E> listValue = active.get(key);
		if (listValue != null) {
			listValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			if (listValue.valueType == GValueType.PutList) {
				return listValue;
			} else if (listValue.valueType == GValueType.Delete) {
				return listValue;
			}
			reverseOrderList.add(listValue);
		}
		Iterator<SegmentKList<K, E>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKList<K, E> inactive = iterator.next();
			listValue = inactive.get(key);
			if (listValue != null) {
				listValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				reverseOrderList.add(listValue);
				if (listValue.valueType == GValueType.PutList || listValue.valueType == GValueType.Delete) {
					break;
				}
			}
		}
		return reverseOrderList.size() == 0 ? null : mergeGSValueList(reverseOrderList);
	}

	private GSValueList<E> mergeGSValueList(List<GSValueList<E>> reverseOrderList) {
		if (reverseOrderList == null || reverseOrderList.size() == 0) {
			return null;
		}
		//TODO check TTL
		int index = reverseOrderList.size() - 1;
		//Value list is right order.
		List<GSValue<E>> newList = new ArrayList<>();
		//judge a compacted map value's type is a little complicated.
		// any -> Delete -> AddList -> AddList, final type must be PutList
		// any -> PutList -> AddList -> AddList, final type must be PutList
		// AddList -> AddList -> AddList, final type must be AddList
		GValueType finalGValueType = GValueType.AddList;
		//map value's seqID is newest. that will be more confused.
		long seqID = reverseOrderList.get(0).getSeqID();
		while (index >= 0) {
			GSValueList<E> currentList = reverseOrderList.get(index);
			if (currentList.getValueType() == GValueType.Delete) {
				newList.clear();
				finalGValueType = GValueType.PutList;
			} else if (currentList.getValueType() == GValueType.PutList) {
				newList.clear();
				newList.addAll(currentList.value);
				finalGValueType = GValueType.PutList;
			} else {
				newList.addAll(currentList.value);
			}
			index--;
		}

		return new GSValueList<>(newList, finalGValueType, seqID);
	}

	@Override
	public void getAll(Map<K, GSValue<List<GSValue<E>>>> container) {
		//TODO
	}

	@Override
	public void allKeysIncludeDeleted(Set<K> container) {
		Segment activeMap = getActiveSegment();
		container.addAll(activeMap.getData().keySet());

		Iterator<SegmentKList<K, E>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKList<K, E> inactive = iterator.next();
			container.addAll(inactive.getData().keySet());
		}
	}

	@Override
	public void removeKey(K key) {
		active.removeKey(key);
		checkResource();
	}

	@Override
	public void reset() {
		active = new SegmentKListImpl<>(segmentID++, gRegionContext);
	}

	@Override
	public GValueType contains(K key) {
		// TODO: maybe return wrong answer, by pass to region.
		GSValueList<E> listValue = active.get(key);
		if (listValue != null) {
			listValue.requestCount++;
			gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
			return listValue.valueType;
		}
		Iterator<SegmentKList<K, E>> iterator = snapshotQueue.descendingIterator();
		while (iterator.hasNext()) {
			SegmentKList<K, E> inactive = iterator.next();
			listValue = inactive.get(key);
			if (listValue != null) {
				listValue.requestCount++;
				gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferHitCount();
				return listValue.valueType;
			}
		}
		gRegionContext.getGContext().getSupervisor().getCacheManager().getCacheStats().addWriteBufferMissCount();
		return null;
	}
}
