/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.memstore;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * stats of writerBuffer, including the number of page, and it's average size, etc.
 * each region has its own WriteBufferStats.
 */
public class WriteBufferStats {
	private volatile int averageKeyLen = -2;
	private volatile int averageValueLen = -1;
	private AtomicInteger totalRecordCount = new AtomicInteger(0);
	private AtomicInteger totalFlushingRecordCount = new AtomicInteger(0);
	private AtomicInteger flushingSegmentCount = new AtomicInteger(0);
	private final WriteBufferManager writeBufferManager;

	public WriteBufferStats(WriteBufferManager writeBufferManager) {
		this.writeBufferManager = writeBufferManager;
	}

	public int getAverageKVSize() {
		return averageKeyLen + averageValueLen;
	}

	public int getAverageKeyLen() {
		return averageKeyLen;
	}

	public int getAverageValueLen() {
		return averageValueLen;
	}

	public int getTotalRecordCount() {
		return totalRecordCount.get();
	}

	public int getFlushingSegmentCount() {
		return flushingSegmentCount.get();
	}

	public void setAverageKeyLen(int averageKeyLen) {
		this.averageKeyLen = averageKeyLen;
	}

	public void setAverageValueLen(int averageValueLen) {
		this.averageValueLen = averageValueLen;
	}

	public void addTotalRecordCount(int totalRecordCount) {
		this.totalRecordCount.addAndGet(totalRecordCount);
		this.writeBufferManager.addTotalRecordCount(totalRecordCount);
	}

	public void addTotalFlushingRecordCount(int totalRecordCount) {
		this.totalFlushingRecordCount.addAndGet(totalRecordCount);
		this.writeBufferManager.addTotalFlushingRecordCount(totalRecordCount);
	}

	public void addFlushingSegmentCount(int flushingSegmentCount) {
		this.flushingSegmentCount.addAndGet(flushingSegmentCount);
		this.writeBufferManager.addTotalFlushingSegmentCount(flushingSegmentCount);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("averageKeyLen", averageKeyLen).
			add("averageValueLen", averageValueLen).
			add("totalRecordCount", totalRecordCount).
			add("totalFlushingRecordCount", totalFlushingRecordCount).
			add("flushingSegmentCount", flushingSegmentCount).toString();
	}
}
