/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.metrics;

import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.gemini.engine.filecache.FileCacheStat;

/**
 * Metrics for file cache.
 */
public class FileCacheMetrics extends MetricsBase {

	private static final String HIT_RATIO = "hit_ratio";

	private static final String AVG_LOCAL_WRITE_SIZE = "avg_local_write_size";

	private static final String AVG_LOCAL_WRITE_TIME = "avg_local_write_time";

	private static final String AVG_LOCAL_READ_SIZE = "avg_local_read_size";

	private static final String AVG_LOCAL_READ_TIME = "avg_local_read_time";

	private static final String AVG_DFS_WRITE_SIZE = "avg_dfs_write_size";

	private static final String AVG_DFS_WRITE_TIME = "avg_dfs_write_time";

	private static final String AVG_DFS_READ_SIZE = "avg_dfs_read_size";

	private static final String AVG_DFS_READ_TIME = "avg_dfs_read_time";

	private static final String AVG_COMPRESS_RATIO = "avg_compress_ratio";

	private boolean hasRegistered;

	public FileCacheMetrics(MetricGroup metricGroup, int sampleCountInterval) {
		this(metricGroup, sampleCountInterval, DEFAULT_HISTOGRAM_WINDOW_SIZE);
	}

	public FileCacheMetrics(MetricGroup metricGroup, int sampleCountInterval, int histogramWindowSize) {
		super(metricGroup, sampleCountInterval, histogramWindowSize);
		this.hasRegistered = false;
	}

	public void register(FileCacheStat fileCacheStat) {
		synchronized (this) {
			if (hasRegistered) {
				return;
			}
			hasRegistered = true;
		}

		getGaugeMetric(HIT_RATIO, () -> {
			double hit = fileCacheStat.hitCount.get();
			double total = fileCacheStat.missCount.get() + hit;
			return total == 0 ? 100 : hit / total * 100;
		});

		getGaugeMetric(AVG_LOCAL_WRITE_SIZE, () -> {
			double total = fileCacheStat.totalLocalWriteCount.get();
			double size = fileCacheStat.totalLocalWriteSize.get();
			return total == 0 ? 0 : size / total;
		});

		getGaugeMetric(AVG_COMPRESS_RATIO, () -> {
			double total = fileCacheStat.totalLocalOriDataSize.get();
			double compressed = fileCacheStat.totalLocalWriteCount.get();
			return compressed == 0 ? 0 : total / compressed;
		});

		getGaugeMetric(AVG_LOCAL_WRITE_TIME, () -> {
			double total = fileCacheStat.totalLocalWriteCount.get();
			double time = fileCacheStat.totalLocalWriteTime.get();
			return total == 0 ? 0 : time / total;
		});

		getGaugeMetric(AVG_LOCAL_READ_SIZE, () -> {
			double total = fileCacheStat.totalLocalReadCount.get();
			double size = fileCacheStat.totalLocalReadSize.get();
			return total == 0 ? 0 : size / total;
		});

		getGaugeMetric(AVG_LOCAL_READ_TIME, () -> {
			double total = fileCacheStat.totalLocalReadCount.get();
			double time = fileCacheStat.totalLocalReadTime.get();
			return total == 0 ? 0 : time / total;
		});

		getGaugeMetric(AVG_DFS_WRITE_SIZE, () -> {
			double total = fileCacheStat.totalDFSWriteCount.get();
			double size = fileCacheStat.totalDFSWriteSize.get();
			return total == 0 ? 0 : size / total;
		});

		getGaugeMetric(AVG_DFS_WRITE_TIME, () -> {
			double total = fileCacheStat.totalDFSWriteCount.get();
			double time = fileCacheStat.totalDFSWriteTime.get();
			return total == 0 ? 0 : time / total;
		});

		getGaugeMetric(AVG_DFS_READ_SIZE, () -> {
			double total = fileCacheStat.totalDFSReadCount.get();
			double size = fileCacheStat.totalDFSReadSize.get();
			return total == 0 ? 0 : size / total;
		});

		getGaugeMetric(AVG_DFS_READ_TIME, () -> {
			double total = fileCacheStat.totalDFSReadCount.get();
			double time = fileCacheStat.totalDFSReadTime.get();
			return total == 0 ? 0 : time / total;
		});
	}
}
