/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.metrics;

import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.gemini.engine.fs.FileManagerStat;

/**
 * FileManagerMetrics.
 */
public class FileManagerMetrics extends MetricsBase {

	private static final String MAX_USED = "max_used_file";

	private static final String USED = "used_file";

	private static final String WAITING_DB_DELETION = "waiting_db_deletion";

	private static final String WAITING_SNAPSHOT_DELETION = "waiting_snapshot_deletion";

	private static final String MARK_DELETION_FILE = "mark_deletion_file";

	private static final String TOTAL_CREATED_FILE = "total_created_file";

	private static final String TOTAL_DATA_SIZE = "total_data_size";

	private boolean hasRegistered;

	public FileManagerMetrics(
		MetricGroup metricGroup,
		int sampleCountInterval) {
		this(metricGroup, sampleCountInterval, DEFAULT_HISTOGRAM_WINDOW_SIZE);
	}

	public FileManagerMetrics(
		MetricGroup metricGroup,
		int sampleCountInterval,
		int histogramWindowSize) {
		super(metricGroup, sampleCountInterval, histogramWindowSize);
		this.hasRegistered = false;
	}

	public void register(FileManagerStat fileManagerStat) {
		synchronized (this) {
			if (hasRegistered) {
				return;
			}
			hasRegistered = true;
		}

		getGaugeMetric(MAX_USED, () -> fileManagerStat.maxUsedFile.get());
		getGaugeMetric(TOTAL_CREATED_FILE, () -> fileManagerStat.totalCreatedFile.get());
		getGaugeMetric(TOTAL_DATA_SIZE, () -> fileManagerStat.totalDataSize.get());
	}

	public void registerUsedFile(Gauge<Integer> gauge) {
		getGaugeMetric(USED, gauge);
	}

	public void registerWaitingDbDeletion(Gauge<Integer> gauge) {
		getGaugeMetric(WAITING_DB_DELETION, gauge);
	}

	public void registerWaitingSnapshotDeletion(Gauge<Integer> gauge) {
		getGaugeMetric(WAITING_SNAPSHOT_DELETION, gauge);
	}

	public void registerMarkDeletionFile(Gauge<Integer> gauge) {
		getGaugeMetric(MARK_DELETION_FILE, gauge);
	}
}
