/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.metrics;

import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.Histogram;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.metrics.SimpleCounter;
import org.apache.flink.runtime.metrics.SimpleHistogram;
import org.apache.flink.util.Preconditions;

import java.util.HashMap;
import java.util.Map;

/**
 * base metrics.
 */
public class MetricsBase {

	protected final Map<String, Histogram> histogramMetrics = new HashMap<>();

	protected final Map<String, Gauge> gaugeMetrics = new HashMap<>();

	protected final Map<String, Counter> counterMetrics = new HashMap<>();

	protected final MetricGroup metricGroup;

	protected final int sampleCountInterval;

	protected final boolean metricsSampleEnabled;

	protected static final int DEFAULT_HISTOGRAM_WINDOW_SIZE = 5;

	protected final int histogramWindowSize;

	MetricsBase(MetricGroup metricGroup, int sampleCountInterval) {
		this(metricGroup, sampleCountInterval, DEFAULT_HISTOGRAM_WINDOW_SIZE);
	}

	MetricsBase(MetricGroup metricGroup, int sampleCountInterval, int histogramWindowSize) {
		this.metricGroup = Preconditions.checkNotNull(metricGroup);
		this.sampleCountInterval = sampleCountInterval;
		this.metricsSampleEnabled = sampleCountInterval > 1;

		Preconditions.checkArgument(histogramWindowSize > 0);
		this.histogramWindowSize = histogramWindowSize;
	}

	public MetricGroup getMetricGroup() {
		return metricGroup;
	}

	public void registerMetricsCacheStat(MetricsRegisterAble metricsRegisterAble) {
		metricsRegisterAble.registerMetrics(this.metricGroup);
	}

	public boolean isMetricsSampleEnabled() {
		return metricsSampleEnabled;
	}

	public int getSampleCountInterval() {
		return sampleCountInterval;
	}

	public Histogram getHistogramMetric(String metricName) {
		return getHistogramMetric(metricName, new SimpleHistogram(histogramWindowSize));
	}

	public Histogram getHistogramMetric(String metricName, Histogram histogram) {
		Histogram histogramMetric = histogramMetrics.get(metricName);
		if (histogramMetric == null) {
			histogramMetric = histogram;
			metricGroup.histogram(metricName, histogramMetric);
			histogramMetrics.put(metricName, histogramMetric);
		}
		return histogramMetric;
	}

	public SimpleGauge<Long> getGaugeMetric(String metricName) {
		return (SimpleGauge<Long>) getGaugeMetric(metricName, new SimpleGauge<Long>());
	}

	@SuppressWarnings("unchecked")
	public <T> Gauge<T> getGaugeMetric(String metricName, Gauge<T> gauge) {
		Gauge gaugeMetric = gaugeMetrics.get(metricName);
		if (gaugeMetric == null) {
			gaugeMetric = gauge;
			metricGroup.gauge(metricName, gaugeMetric);
			gaugeMetrics.put(metricName, gaugeMetric);
		}
		return (Gauge<T>) gaugeMetric;
	}

	public Counter getCounterMetric(String metricName) {
		return getCounterMetric(metricName, new SimpleCounter());
	}

	public Counter getCounterMetric(String metricName, Counter counter) {
		Counter counterMetric = counterMetrics.get(metricName);
		if (counterMetric == null) {
			counterMetric = counter;
			metricGroup.counter(metricName, counterMetric);
			counterMetrics.put(metricName, counterMetric);
		}
		return counterMetric;
	}

	static class SimpleGauge<T> implements Gauge<T> {

		private volatile T value;

		public void updateValue(T value) {
			this.value = value;
		}

		@Override
		public T getValue() {
			return value;
		}
	}
}
