/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.fs.FileWriter;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap;
import org.apache.flink.runtime.state.gemini.engine.page.compress.GCompressAlgorithm;
import org.apache.flink.runtime.state.gemini.engine.rm.Allocator;
import org.apache.flink.runtime.state.gemini.engine.rm.ReferenceCount.ReleaseType;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

/**
 * DataPage.
 */
public interface DataPage<K, V> {
	void addReferenceCount();

	void delReferenceCount(ReleaseType releaseType);

	int getCheckSum();

	/**
	 * DataPageType.
	 */
	enum DataPageType {//KV.
		KV((byte) 1), KHashMap((byte) 2), KSortedMap((byte) 3), KList((byte) 4);

		private final byte code;

		DataPageType(final byte c) {
			this.code = c;
		}

		public byte getCode() {
			return this.code;
		}

		public static DataPageType valueOf(byte pageType) {
			switch (pageType) {
				case 1:
					return KV;
				case 2:
					return KHashMap;
				case 3:
					return KSortedMap;
				case 4:
					return KList;
				default:
					throw new GeminiRuntimeException("Error PageType:" + pageType);
			}
		}}

	BinaryValue getBinaryValue(K key);

	GSValue<V> get(K key);

	boolean contains(K key);

	long getVersion();

	DataPageType getDataPageType();

	int getSize();

	int getCount();

	Map<K, GSValue<V>> getPOJOMap();

	Set<K> getPOJOSet();

	Tuple2<DataPage, DataPage> split(
		int curBucketNum, int curIndex, Allocator allocator, GCompressAlgorithm gCompressAlgorithm);

	int write(
		FileWriter writer,
		PageSerdeFlink<K, V> pageSerdeFlink,
		PageAddress pageAddress,
		GCompressAlgorithm gCompressAlgorithm, boolean checksumEnable) throws IOException;

	long getCompactionCount();

	GBinaryHashMap<K> getGBinaryHashMap();
}
