/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryList;
import org.apache.flink.runtime.state.gemini.engine.rm.Allocator;
import org.apache.flink.runtime.state.gemini.engine.rm.GByteBuffer;
import org.apache.flink.runtime.state.gemini.engine.utils.SeqIDUtils;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import static org.apache.flink.runtime.state.gemini.engine.page.DataPageKMapImpl.getDuplicateBB;
import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap.EMPTY_G_BINARY_HASHMAP;
import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryList.EMPTY_G_BINARY_LIST;

/**
 * DataPageKListImpl.
 */
public class DataPageKListImpl<K, E> extends DataPageKVImpl<K, List<GSValue<E>>> implements DataPageKList<K, E> {

	private final TypeSerializer<E> originValueTypeSerializer;

	public DataPageKListImpl(
		GBinaryHashMap<K> gBinaryHashMap,
		TypeSerializer<E> valueTypeSerializer,
		GListValueTypeSerializer<E> gListValueTypeSerializer) {
		super(gBinaryHashMap, gListValueTypeSerializer);
		this.originValueTypeSerializer = valueTypeSerializer;
	}

	@Override
	public DataPageType getDataPageType() {
		return DataPageType.KList;
	}

	public static <E> BinaryValue doCompactionKListValue(
		List<BinaryValue> valueByOrder,
		TypeSerializer<E> mkTypeSerializer,
		boolean isMajor,
		long version,
		int logicPageId,
		Allocator allocator) {

		try {
			List<GBinaryList<E>> listByOrder = new ArrayList<>();
			long seqID = SeqIDUtils.INVALID_SEQID;
			GValueType firstValueType = null;

			for (BinaryValue binaryValue : valueByOrder) {
				if (binaryValue.getgValueType() == GValueType.Delete) {
					firstValueType = GValueType.Delete;
					continue;
				}

				GBinaryList<E> gBinaryList = new GBinaryList<>(getDuplicateBB(binaryValue).getByteBuffer(), mkTypeSerializer);
				//pick up newest page's seqID.
				seqID = Math.max(seqID, binaryValue.getSeqID());
				listByOrder.add(gBinaryList);
				if (firstValueType == null) {
					firstValueType = binaryValue.getgValueType();
				}
			}

			GBinaryList<E> gBinaryListMerge = GBinaryList.mergeGBinaryList(listByOrder, mkTypeSerializer, allocator);

			ByteBuffer bb = gBinaryListMerge == EMPTY_G_BINARY_LIST ? null : gBinaryListMerge.getData();
			GValueType gValueType = judgeFinalValueType(bb, firstValueType, isMajor);
			return new BinaryValue(bb, gValueType, seqID, 0, gBinaryListMerge.bytesSize());
		} catch (Exception e) {
			throw new GeminiRuntimeException("Internal BUG " + e.getMessage(), e);
		}
	}

	static GValueType judgeFinalValueType(ByteBuffer bb, GValueType firstValueType, boolean isMajor) {
		//judge a compacted map value's type is a little complicated.
		// Delete -> AddList -> AddList,  it only happens when it's minor, and final type can be PutList
		// PutList -> AddList -> AddList, when it's major, it can be PutMap or AddMap, when it's minor, it only can be PutList
		// AddList -> AddList -> AddList, when it's major, it can be PutMap or AddMap, when it's minor, it only can be AddList
		// when compacted map value is null, it only happends when it's major, it can be any one.
		if (bb == null) {
			return GValueType.Delete;
		}

		if (firstValueType == GValueType.Delete) {
			return GValueType.PutList;
		}
		return firstValueType;
	}

	public static <K, E> DataPageKListImpl<K, E> readDataPageKListFrom(
		PageSerdeFlinkListImpl<K, E> pageSerdeFlink, GByteBuffer dataPage, int crc) {
		GBinaryHashMap<K> gBinaryHashMap = new GBinaryHashMap<>(dataPage, pageSerdeFlink.getKeySerde(), crc);
		return new DataPageKListImpl<>(gBinaryHashMap,
			pageSerdeFlink.getValueSerde(),
			pageSerdeFlink.getgListValueTypeSerializer());
	}

	@Override
	public Tuple2<DataPage, DataPage> getSplitDataByGBinaryMap(
		GBinaryHashMap<K> gBinaryHashMap1, GBinaryHashMap<K> gBinaryHashMap2) {
		DataPageKListImpl<K, E> dataPage1 = gBinaryHashMap1 == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKListImpl<>(gBinaryHashMap1,
				originValueTypeSerializer,
				(GListValueTypeSerializer<E>) valueTypeSerializer);
		DataPageKListImpl<K, E> dataPage2 = gBinaryHashMap2 == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKListImpl<>(gBinaryHashMap2,
				originValueTypeSerializer,
				(GListValueTypeSerializer<E>) valueTypeSerializer);
		return Tuple2.of(dataPage1, dataPage2);

	}
}
