/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.ByteBufferDataInputView;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryList;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryList.EMPTY_G_BINARY_LIST;

/**
 * GListValueTypeSerializer for KList.
 */
public final class GListValueTypeSerializer<E> extends AbstractGValueTypeSerializer<List<GSValue<E>>> {

	private final TypeSerializer<E> valueTypeSerializer;

	public GListValueTypeSerializer(TypeSerializer<E> valueTypeSerializer) {
		this.valueTypeSerializer = valueTypeSerializer;
	}

	@Override
	public void serialize(List<GSValue<E>> record, DataOutputView target) throws IOException {
		GBinaryList<E> gBinaryList = GBinaryList.ofGBinaryList(record, this.valueTypeSerializer, allocator);

		if (gBinaryList == EMPTY_G_BINARY_LIST) {
			return;
		}
		byte[] bytes = gBinaryList.getDataByte();
		//Gemini no need to write len.
		target.write(bytes);
	}

	@Override
	public List<GSValue<E>> deserialize(DataInputView source) throws IOException {
		//For Gemini, to save a int indicator when serializing, only use ByteBufferDataInputView
		ByteBufferDataInputView bufferDataInputView = (ByteBufferDataInputView) source;
		int size = bufferDataInputView.available();
		if (size == 0) {
			return new ArrayList<>();
		}
		byte[] bytes = new byte[size];
		source.readFully(bytes);
		GBinaryList<E> gBinaryList = new GBinaryList<>(ByteBuffer.wrap(bytes), this.valueTypeSerializer);
		return gBinaryList.toPOJOList();
	}

}
