/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.ByteBufferDataInputView;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap;
import org.apache.flink.runtime.state.gemini.engine.page.compress.GCompressAlgorithm;
import org.apache.flink.runtime.state.gemini.engine.rm.GUnPooledByteBuffer;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap.EMPTY_G_BINARY_HASHMAP;

/**
 * GMapValueTypeSerialiZer for KMap.
 */
public final class GMapValueTypeSerializer<MK, MV> extends AbstractGMapValueTypeSerializer<MK, MV> {

	public GMapValueTypeSerializer(
		TypeSerializer<MK> mkTypeSerializer,
		TypeSerializer<MV> mvTypeSerializer,
		boolean checksumEnable) {
		super(mkTypeSerializer, mvTypeSerializer, checksumEnable);
	}

	@Override
	public void serialize(Map<MK, GSValue<MV>> record, DataOutputView target) throws IOException {
		List<Tuple2<MK, GSValue<MV>>> keyValueList = record.entrySet().stream().map(entry -> new Tuple2<>(entry.getKey(),
			entry.getValue())).collect(Collectors.toList());
		//for value Page, version and logicPageId of 'page' is ignored.
		GBinaryHashMap<MK> gBinaryHashMap = GBinaryHashMap.of(DataPage.DataPageType.KV,
			keyValueList,
			mkTypeSerializer,
			mvTypeSerializer,
			-1,
			-1,
			allocator,
			1,
			//signle value, no need compress
			GCompressAlgorithm.None);

		if (gBinaryHashMap == EMPTY_G_BINARY_HASHMAP) {
			return;
		}

		byte[] bytes = gBinaryHashMap.getDataByte(this.checksumEnable);
		//Gemini no need to write len.
		target.write(bytes);
	}

	@Override
	public Map<MK, GSValue<MV>> deserialize(DataInputView source) throws IOException {
		//For Gemini, to save a int indicator when serializing, only use ByteBufferDataInputView
		ByteBufferDataInputView bufferDataInputView = (ByteBufferDataInputView) source;
		int size = bufferDataInputView.available();
		if (size == 0) {
			return new HashMap<>(0);
		}
		byte[] bytes = new byte[size];
		source.readFully(bytes);
		GBinaryHashMap<MK> gBinaryHashMap = new GBinaryHashMap<>(new GUnPooledByteBuffer(ByteBuffer.wrap(bytes)), mkTypeSerializer);
		return gBinaryHashMap.toPOJOMap(mvTypeSerializer);
	}
}
