/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.ByteBufferDataInputView;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinarySortedMap;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GComparator;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinarySortedMap.EMPTY_G_BINARY_SORTEDMAP;

/**
 * GSortedMapValueTypeSerialiZer for KMap.
 */
public final class GSortedMapValueTypeSerialiZer<MK, MV> extends AbstractGMapValueTypeSerializer<MK, MV> {

	private final GComparator<MK> gComparator;

	public GSortedMapValueTypeSerialiZer(
		TypeSerializer<MK> mkTypeSerializer,
		TypeSerializer<MV> mvTypeSerializer,
		GComparator<MK> gComparator,
		boolean checksumEnable) {
		super(mkTypeSerializer, mvTypeSerializer, checksumEnable);
		this.gComparator = gComparator;
	}

	@Override
	public void serialize(Map<MK, GSValue<MV>> record, DataOutputView target) throws IOException {
		List<Tuple2<MK, GSValue<MV>>> keyValueList = record.entrySet().stream().map(entry -> new Tuple2<>(entry.getKey(),
			entry.getValue())).collect(Collectors.toList());
		//for value Page, version and logicPageId of 'page' is ignored.
		GBinarySortedMap<MK> gBinarySortedMap = GBinarySortedMap.of(DataPage.DataPageType.KV,
			keyValueList,
			mkTypeSerializer,
			mvTypeSerializer,
			this.gComparator,
			-1,
			-1,
			allocator,
			1);

		if (gBinarySortedMap == EMPTY_G_BINARY_SORTEDMAP) {
			return;
		}

		byte[] bytes = gBinarySortedMap.getDataByte(this.checksumEnable);
		//Gemini no need to write len.
		target.write(bytes);
	}

	@Override
	public Map<MK, GSValue<MV>> deserialize(DataInputView source) throws IOException {
		//For Gemini, to save a int indicator when serializing, only use ByteBufferDataInputView
		Map<MK, GSValue<MV>> result = new TreeMap<>(gComparator.getJDKCompactor());
		ByteBufferDataInputView bufferDataInputView = (ByteBufferDataInputView) source;
		int size = bufferDataInputView.available();
		if (size == 0) {
			return result;
		}
		byte[] bytes = new byte[size];
		source.readFully(bytes);
		GBinarySortedMap<MK> gBinarySortedMap = new GBinarySortedMap<>(ByteBuffer.wrap(bytes),
			mkTypeSerializer,
			gComparator);
		return gBinarySortedMap.getgBinaryHashMap().toPOJOMap(result, mvTypeSerializer);
	}
}
