/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.CompatibilityResult;
import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.common.typeutils.TypeSerializerConfigSnapshot;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.state.gemini.engine.GeminiPKey2;

import java.io.IOException;
import java.util.Objects;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * Serializer for GeminiPKey2.
 * @param <F>
 * @param <S>
 */
public class PKey2Serializer<F, S> extends TypeSerializer<GeminiPKey2<F, S>> {
	private final TypeSerializer<F> firstSerializer;
	private final TypeSerializer<S> secondSerializer;

	public PKey2Serializer(
		TypeSerializer<F> firstSerializer,
		TypeSerializer<S> secondSerializer) {
		this.firstSerializer = checkNotNull(firstSerializer);
		this.secondSerializer = checkNotNull(secondSerializer);
	}

	@Override
	public boolean isImmutableType() {
		return true;
	}

	@Override
	public TypeSerializer<GeminiPKey2<F, S>> duplicate() {
		throw new UnsupportedOperationException();
	}

	@Override
	public GeminiPKey2<F, S> createInstance() {
		return new GeminiPKey2<>(
			firstSerializer.createInstance(),
			secondSerializer.createInstance());
	}

	@Override
	public GeminiPKey2<F, S> copy(GeminiPKey2<F, S> from) {
		return new GeminiPKey2<>(
			firstSerializer.copy(from.getFirstKey()),
			secondSerializer.copy(from.getSecondKey()));
	}

	@Override
	public GeminiPKey2<F, S> copy(GeminiPKey2<F, S> from, GeminiPKey2<F, S> reuse) {
		return copy(from);
	}

	@Override
	public int getLength() {
		return 0;
	}

	@Override
	public void serialize(GeminiPKey2<F, S> record, DataOutputView target) throws IOException {
		firstSerializer.serialize(record.getFirstKey(), target);
		secondSerializer.serialize(record.getSecondKey(), target);
	}

	@Override
	public GeminiPKey2<F, S> deserialize(DataInputView source) throws IOException {
		return new GeminiPKey2<>(firstSerializer.deserialize(source), secondSerializer.deserialize(source));
	}

	@Override
	public GeminiPKey2<F, S> deserialize(GeminiPKey2<F, S> reuse, DataInputView source) throws IOException {
		return deserialize(source);
	}

	@Override
	public void copy(DataInputView source, DataOutputView target) throws IOException {
		firstSerializer.copy(source, target);
		secondSerializer.copy(source, target);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if (obj == null || obj.getClass() != getClass()) {
			return false;
		}

		PKey2Serializer<F, S> other = (PKey2Serializer<F, S>) obj;

		return other.firstSerializer.equals(firstSerializer) &&
			other.secondSerializer.equals(secondSerializer);
	}

	@Override
	public boolean canEqual(Object obj) {
		return obj instanceof PKey2Serializer;
	}

	@Override
	public int hashCode() {
		return Objects.hash(firstSerializer, secondSerializer);
	}

	@Override
	public TypeSerializerConfigSnapshot snapshotConfiguration() {
		throw new UnsupportedOperationException();
	}

	@Override
	public CompatibilityResult<GeminiPKey2<F, S>> ensureCompatibility(TypeSerializerConfigSnapshot configSnapshot) {
		throw new UnsupportedOperationException();
	}

	public TypeSerializer<F> getFirstSerializer() {
		return firstSerializer;
	}

	public TypeSerializer<S> getSecondSerializer() {
		return secondSerializer;
	}
}

