/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import java.util.Objects;

/**
 * PageIndexContextHashImpl.
 */
public class PageIndexContextHashImpl implements PageIndexContext {
	private final int curBucketNum;
	private final int curIndex;
	private final LogicChainedPage pageID;
	private final boolean needSplit;

	public PageIndexContextHashImpl(
		int curBucketNum, int curIndex, LogicChainedPage pageID, boolean needSplit) {
		this.curBucketNum = curBucketNum;
		this.curIndex = curIndex;
		this.pageID = pageID;
		this.needSplit = needSplit;
	}

	public static PageIndexContext of(int curBucketNum, int curIndex, LogicChainedPage pageID, boolean needSplit) {
		return new PageIndexContextHashImpl(curBucketNum, curIndex, pageID, needSplit);
	}

	public static PageIndexContext of(PageIndexContextHashImpl pageIndexContext, LogicChainedPage logicChainedPage) {
		return new PageIndexContextHashImpl(pageIndexContext.curBucketNum,
			pageIndexContext.curIndex,
			logicChainedPage,
			false);
	}

	@Override
	public LogicChainedPage getPageID() {
		return pageID;
	}

	@Override
	public int getPageIndexID() {
		return this.curIndex;
	}

	@Override
	public boolean isNeedSplit() {
		return needSplit;
	}

	public int getCurBucketNum() {
		return curBucketNum;
	}

	public int getCurIndex() {
		return curIndex;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		PageIndexContextHashImpl that = (PageIndexContextHashImpl) o;
		return curBucketNum == that.curBucketNum && curIndex == that.curIndex && needSplit == that.needSplit && Objects.equals(
			pageID,
			that.pageID);
	}

	@Override
	public int hashCode() {
		return Objects.hash(curBucketNum, curIndex, pageID, needSplit);
	}
}
