/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GComparator;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GComparatorBytesImpl;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GComparatorType;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GComparatorUserImpl;

import java.util.Comparator;

/**
 * PageSerdeFlink2KeyImpl.
 */
public class PageSerdeFlink2KeyImpl<K1, K2, V> extends PageSerdeFlinkImpl<K1, V> implements PageSerdeFlink2Key<K1, K2, V> {

	private final TypeSerializer<K2> key2Serde;
	private final GComparator<K2> mapComparator;
	private final AbstractGMapValueTypeSerializer<K2, V> gMapValueTypeSerialiZer;

	public PageSerdeFlink2KeyImpl(
		TypeSerializer<K1> keySerde,
		TypeSerializer<K2> key2Serde,
		TypeSerializer<V> valueSerce,
		GComparator<K2> mapComparator,
		boolean checksumEnable) {
		super(keySerde, valueSerce);
		this.key2Serde = key2Serde;
		this.mapComparator = mapComparator;
		this.gMapValueTypeSerialiZer = mapComparator == null ? new GMapValueTypeSerializer<>(key2Serde,
			valueSerce,
			checksumEnable) : new GSortedMapValueTypeSerialiZer<>(key2Serde, valueSerce, mapComparator, checksumEnable);
	}

	@Override
	public TypeSerializer<K2> getKey2Serde() {
		return this.key2Serde;
	}

	@Override
	public GComparator<K2> getMapComparator() {
		return this.mapComparator;
	}

	@Override
	public AbstractGMapValueTypeSerializer<K2, V> getMapValueTypeSerializer() {
		return this.gMapValueTypeSerialiZer;
	}

	public static <K, MK, MV> PageSerdeFlink2KeyImpl<K, MK, MV> of(
		TypeSerializer<K> keySerializer,
		TypeSerializer<MK> mapKeySerializer,
		TypeSerializer<MV> mapValueSerializer,
		Comparator<MK> userComparator,
		GComparatorType gComparatorType,
		boolean checksumEnable) {
		GComparator<MK> mapComparator = userComparator == null
			? null
			: buildGComparator(userComparator, gComparatorType, mapKeySerializer);
		return new PageSerdeFlink2KeyImpl<>(keySerializer,
			mapKeySerializer,
			mapValueSerializer,
			mapComparator,
			checksumEnable);
	}

	private static <MK> GComparator<MK> buildGComparator(
		Comparator<MK> userComparator, GComparatorType gComparatorType, TypeSerializer<MK> mapKeySerializer) {
		switch (gComparatorType) {
			case bytes:
				return new GComparatorBytesImpl<>(userComparator, mapKeySerializer);
			case user:
				return new GComparatorUserImpl<>(userComparator, mapKeySerializer);
			default:
				throw new GeminiRuntimeException("invalid GComparatorType:" + gComparatorType);
		}
	}
}
